package org.bouncycastle.pqc.crypto.test;

import java.io.IOException;
import java.security.SecureRandom;
import java.text.ParseException;
import java.util.List;

import junit.framework.TestCase;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.digests.SHA512Digest;
import org.bouncycastle.pqc.crypto.xmss.XMSS;
import org.bouncycastle.pqc.crypto.xmss.XMSSNode;
import org.bouncycastle.pqc.crypto.xmss.XMSSParameters;
import org.bouncycastle.pqc.crypto.xmss.XMSSSignature;
import org.bouncycastle.pqc.crypto.xmss.XMSSUtil;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Hex;

/**
 * Test cases for XMSS class.
 */
public class XMSSTest
    extends TestCase
{

    public void testAuthPath()
    {
        XMSSParameters xmssParams = new XMSSParameters(10, new SHA256Digest());
        XMSS xmss = new XMSS(xmssParams, new NullPRNG());
        xmss.generateKeys();
        String[] expectedAuthPath = {
            "e0e81477f50e2b8719d1d9a3c1a55d5460711b88eb4f2c8df8d17d38832de9d4",
            "251142aa2cdb5a9d81a0c7a98e2c3c95bf96e73c68c78d467186eeac1dffce38",
            "53a6a544a1331bf650758d64bb8fab1104170824894fdd8a284d281aec9d6270",
            "5683d8f8de11a5845b9d1b5cedb7fd0bd76da3341e75ddd05f261520228664d0",
            "16b265f881e4b7bd303b7c38b8adcf31a7c211bcddb5260080105c1b70c41ee9",
            "9d421eefaf1e51d762f245853cc36250bfbb3f547770303dc8a38d03dec2d0a7",
            "233b634f83a2456f279126c7fcb47f9301cb2d5e8db69f4721f68000d78cffb3",
            "9317371748b85e824bd82284875b7bb65604c94ff0b8816cef677ade31963bd5",
            "14ba59e3d998735a963d018b840a80be11a52ccdadaceb7ed45f0763e6cd49ca",
            "6af97f6bde85e4569178192bab4fd59d8c2b1fec181f27a8fa2c17552c16e3b5"
        };
        byte[] signature = xmss.sign(new byte[1024]);
        XMSSSignature sig = new XMSSSignature.Builder(xmssParams).withSignature(signature).build();

        List<XMSSNode> authPath = sig.getAuthPath();
        for (int i = 0; i < authPath.size(); i++)
        {
            assertEquals(expectedAuthPath[i], Hex.toHexString(authPath.get(i).getValue()));
        }
    }

    public void testGenKeyPairSHA256()
    {
        XMSSParameters xmssParams = new XMSSParameters(10, new SHA256Digest());
        XMSS xmss = new XMSS(xmssParams, new NullPRNG());
        xmss.generateKeys();
        byte[] privateKey = xmss.exportPrivateKey();
        byte[] publicKey = xmss.exportPublicKey();
        String expectedPrivateKey = "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000073c3fc6de1195aa5d69f9dafc9db8504aa8059115e8319ca15cf58a1c83c0de3";
        String expectedPublicKey = "73c3fc6de1195aa5d69f9dafc9db8504aa8059115e8319ca15cf58a1c83c0de30000000000000000000000000000000000000000000000000000000000000000";
        byte[] strippedPrivateKey = XMSSUtil.extractBytesAtOffset(privateKey, 0, (Hex.decode(expectedPrivateKey).length));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedPrivateKey), strippedPrivateKey));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedPublicKey), publicKey));
    }

    public void testGenKeyPairSHA512()
    {
        XMSSParameters xmssParams = new XMSSParameters(10, new SHA512Digest());
        XMSS xmss = new XMSS(xmssParams, new NullPRNG());
        xmss.generateKeys();
        byte[] privateKey = xmss.exportPrivateKey();
        byte[] publicKey = xmss.exportPublicKey();
        String expectedPrivateKey = "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000f48df90f8e217076d8af6327691321bdcf63668c4bd28d021d49f2334eca845fa3073991049286c0eef5dc7f23ec0b31f5c1bd1e5b8edb2403ae02f292f6f30e";
        String expectedPublicKey = "f48df90f8e217076d8af6327691321bdcf63668c4bd28d021d49f2334eca845fa3073991049286c0eef5dc7f23ec0b31f5c1bd1e5b8edb2403ae02f292f6f30e00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
        byte[] strippedPrivateKey = XMSSUtil.extractBytesAtOffset(privateKey, 0, (Hex.decode(expectedPrivateKey).length));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedPrivateKey), strippedPrivateKey));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedPublicKey), publicKey));
    }

    public void testSignSHA256()
    {
        XMSSParameters params = new XMSSParameters(10, new SHA256Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        byte[] message = new byte[1024];
        byte[] sig1 = xmss.sign(message);
        byte[] sig2 = xmss.sign(message);
        byte[] sig3 = xmss.sign(message);
        String expectedSig1 = "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";
        String expectedSig2 = "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";
        String expectedSig3 = "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";
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig1), sig1));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig2), sig2));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig3), sig3));
    }

    public void testSignSHA256CompleteEvenHeight1()
    {
        final String[] signatures = {
            "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",
            "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",
            "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",
            "0000000386edf8472409c6a0fe2f3da07457868477f3f745d6717aec4a4ba8c7ab1647486e381abd8cbdf2430a1be34c7dbb65422fb1f261079d086b4212eea4fc20e39638c6098e1fe17f4df9c51ff96f6db6c6f7488f167cdd09f52b191a85782ca73e42d7800af123d2ac2d705844b093404f2072dd430f48bdb09438a21119e1b3730ad685a39c4e8c67750bf56795a43d93be6d5d6181d05922a13aa08d95fba85b6653716610269ce617146ff16b1b781fe1d512b1bac6705ba24cab5e5389ffbc038a751e77edc1a927d33fb3449cc2a639ae6052b59f5a868b370fa8778e939a30149264898b8c050a093e1d345f3076feb8c50c080a2e0604c45995c7482c23a12400a3318a18d8fc93668acfa68a45c8797e3749bb221a3584f97445a6c6ad215be5a12e582d3fdfd2eb5f74032a6e7aa35718389304989170bcbe1c85d36d714bb2fff05f88b85ced919c9bc94dd004804e460818411821e51a6f69731d54acf5e29c8f2090b360fa4afde8008c0e7085d2989badb92df58f45044aa257ecd9857d4f20b3658c49c0a09b2a53ec1451a5e3a8e4d60432504475992c2099d3866bdafaa6cc626f31e95f20a00d7870a74b957c2db3b93544094c2db82f89cddc95724b541840a32d6ca4465e335c4bc8fdf5aa5ebdc3b3ad9874954771a7340530ab22c6120b65c45bc8c28b3e159bd8511a7e1898c6bd83340ced8c1dae51030a0159a596f08cb761e777e19bf31d9492acbfd5c442ac9df91ae90040a93ad5e935270563f453b35c0ddea505343215f0cafafcc125e56fc9ae09f238440a32ef78b942d9b35d06a9c12257925c3aae2761661df4f3df78db246d1526f6f2f9d007614e239e91a8c449601ddaa9516658c78ee08aba6cf01903dcc72e5f9c9699515a1fb1bbe267a8b9fda3000e6ebcee589a4a868695b965f13086c780ee72169894e3da655014eb0a5a3282d37b6b45b7dab67a62a648e29005075555b29b4de95eeb3656e71807feb351eeb20ddf3757ed8de48784c75f24b772fc8440b1d0fa18f8a747c87b60cf5ae0de49edc1dbcca5fed608f93a8b45cad33e444aaa0069586bf94e5a4546696dd37ad6b1a3835a250f6495424a1adb50a9e436c0bf29623e5b09bbe1d67e87ba0c6f3a2f92a4545cc9b6668c0609dd5149c52f088ede4f9ffcd8624bc6a46996d9a5b78c5533249637996f1ea03c949b27e5434b3b4df286a35c2a427c4f37fb48218cd4f75673e6c902071a289ba3d69de281833193b83cf2bcbff3746bdeb748d2a89c2c5d4bdbd932346dee04b6881c5abd78994c6ddf1850bd389b84892d0d59c311f67777d6f11f3c9191943db01eaf124556705cea4f2c14457ca2f8f25258ada527901e4daabae095388bcece387f1e363c606e131951d709e0312b01094b52374cf9dbfb1f08e2d7566e0f68a5df93e72bf2fc56c04ea1904166d97e957adcea5efad409cd58dcc9a2fffe2efb3ed1ee5a41fb93138765ab5d80c50bdd8649ae2ae4937153bca2c2271c74dc2cf72b26bdb5d7438bc729d535316ba82c1e7ae7b0041a935d91760f7d80c57517c3bbd6165838a478a9f7a17161f54eb729d8055e9db80be9b4af3f172a54a812e9ed513e8bc6ee2fbf814f8f3ee28bc0570e9aa7d134efd571baeb0e384938487f13722486bacce923136ccd3e3c5cc7c1426ac672fc0b0d53bdee953e5314013e566472a6479776fcd6fa892c6851a9baf4531998851a79f1f839b02953ed6d7de23dc8ed442b031394320e0b97e6693142bd8d3cae99af9848b31c1f290fed2de7ff1d1841c30d3c2d44d71c893f150056c454900c122309a9f588f25889833fac1118122f7c102eb466ac4ec724dbc60d431c470226099956f0a51cf5d78229d91af8e4acbce56d1643650c64ecba2243431c06d4065cf738e5cb0980da72a35425ee889fc27c9d514bc581b7e740ec12b4e75927754a0ae3d61fc3c6a8e5b3c93a748fd4cd93f8bf788879da5b82fc4eea69256ee44b5e8cc1c1bf0bcf577a165b84c2ac3dc885413ab490a4f85eceb5e364b6331e7f11b7a832f974f7facc9a377a69a3c9a4ac69684cc8515557402a5d6f70fcc82c20051c7db5cd3d6d7f3bbaa38ef89f41337ec131ab3e3dc3959b78848f2dc51c9db0fd0fcc3d2bd670d1c2fdf3f681280be64a80508f623cca14fdaebd5cf4374182612c2996f79d4a547afeaf0dcb34a8eba293646d6b76725666dfce8fca65a186532ee1b441a24609c80bb045ffefff2e7a1f330bde5ac665eb7296fdec860b6c885143056d739969107fbe50283076a9e6ee68e92a303cdf6ccb2fd9a7e9a25100e15de3c3beb5562e27e88bb3e02064762052fa3b0e69f70e44dcb1410ac1fc604f27e86796c6d381ebc896a463de474e2949d242d1eac60f4f8d36685392adf7bff0c5d579956ff4e71bc8269233834d56ffe830079a082eba25eadc5ec0252994942119eebfa0817627b1731a791c4fb4586c43dc2190cc25ca783104aeb057cddc69f0b1692254ecf21d1638c84831b536dad113c1f6bca25d3ee8562eb572aaf23614b59ed0e9f8ab077f9c0a022e1f65fbaa56c8c1c64457e584416009587f8f57592f96951ad0620c1bac4350d7260e973f6635062dad70f3135515f5144814ae13356918954a53f90b3980c08da3cba1d8f112ef4992b38d1183638125dd62fa05c68cc3478064fb87406a78335c85866fd427079f647bbe439e7f8235ff966d591a5cee90cfbaac4873a30519db48aa1111b27e930b644c4f83da0f0b6116d9dbce9eb962268dd839441cbfa116f502d46d66008bde12ea1e33689b7b870708a05252d44ebd390380cbf245d6434fd972f0c7d09e6612ef7867b2f39feea4bfb39f255e3abd31677071ed4d8a7f394bb35f29f7465b5efa0d8b0ce9f57619bfe1e7e9beb2e49d1b372fd3c38453e9bd59d04ff5409825280d38e100f29eccc045ecbb69ccf8a0977c2d7a74eb22fab6c20c0f614f725777f9d1c2491436f87b65c032302bc24ee25f321e47b2ef1e0b9901037ca8c061281e1ff6080de1fbd2f41f91f32f0e09710e1c93982cf716daa1c350b4dbbaa17a95a2251cf702fdc87c03e410d9ac0062bbc479ad59262cc6bce4048859718988b62e2c4511ed65f659cf8c579a53a6a544a1331bf650758d64bb8fab1104170824894fdd8a284d281aec9d62705683d8f8de11a5845b9d1b5cedb7fd0bd76da3341e75ddd05f261520228664d0",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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"};
        int height = 4;
        XMSSParameters params = new XMSSParameters(height, new SHA256Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        for (int i = 0; i < (1 << height); i++)
        {
            byte[] signature = xmss.sign(new byte[1024]);
            assertEquals(signatures[i], Hex.toHexString(signature));
        }
        try
        {
            xmss.sign(new byte[1024]);
            fail();
        }
        catch (Exception ex)
        {
        }
    }

    public void testSignSHA256CompleteEvenHeight2()
    {
        final String[] signatures = {
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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",
            "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"};
        int height = 10;
        XMSSParameters params = new XMSSParameters(height, new SHA256Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        for (int i = 0; i < (1 << height); i++)
        {
            byte[] signature = xmss.sign(new byte[1024]);
            switch (i)
            {
            case 0x005b:
                assertEquals(signatures[0], Hex.toHexString(signature));
                break;
            case 0x0822:
                assertEquals(signatures[1], Hex.toHexString(signature));
                break;
            case 0x00b0:
                assertEquals(signatures[2], Hex.toHexString(signature));
                break;
            case 0x016d:
                assertEquals(signatures[3], Hex.toHexString(signature));
                break;
            case 0x0189:
                assertEquals(signatures[4], Hex.toHexString(signature));
                break;
            case 0x01f9:
                assertEquals(signatures[5], Hex.toHexString(signature));
                break;
            case 0x02a1:
                assertEquals(signatures[6], Hex.toHexString(signature));
                break;
            case 0x038b:
                assertEquals(signatures[7], Hex.toHexString(signature));
                break;
            case 0x03fe:
                assertEquals(signatures[8], Hex.toHexString(signature));
                break;
            case 0x03ff:
                assertEquals(signatures[9], Hex.toHexString(signature));
                break;
            }
        }
        try
        {
            xmss.sign(new byte[1024]);
            fail();
        }
        catch (Exception ex)
        {
        }
    }

    public void testSignSHA512()
    {
        XMSSParameters params = new XMSSParameters(10, new SHA512Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        byte[] message = new byte[1024];
        byte[] sig1 = xmss.sign(message);
        byte[] sig2 = xmss.sign(message);
        byte[] sig3 = xmss.sign(message);
        String expectedSig1 = "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";
        String expectedSig2 = "00000001e40f5b447267c994bdc775ce6d98b0e559cba26c4261d70927adf1bb11b3336858fe064267d0514a6d971b69939256b8ce8b3d8ba0d796413f561c6825e720e6c0c15632b943606aad0b427ca04a6808f4e783807111ec1baaded34ef7ec2cf40ba763a4555ae307171ad202f421548e7361dd06b9145e6fbdd34eace57be4829117e9c99bf3bd00dd4e0591c74d4e488e90623c6a4cabd5bc903252eb81eefe8b52b2d9b3df37bcbeb5bcff64203211d1cf9e1bd3250cdcaa429c2ddf2eb1bc7e7d366be2dc0dd774f99937859c40fe04a98ba532744ead3c6199ffe6f740399602658f65bbf67010e99fc57f841b91bc29ccf4b6bed49747b459b9835e4fef8a95cfd67eef5110d71e386a557562cde703369a4380718d957ea7c1377c7230800918136d3e9755bf99941e5e857a4d66478d81a89683de426bb466f9b660a6dd9e4dc28bb25e0afefd311c68355753be9f3c12f85458eede1cf39442c8d51a4f9297a91864cd98ee8ce77584174e594072df3c776dfddaf36cb1c14161965b516230db575b7e2a7306c135a61d6a4e9c8d297bac90828cd6cb55389ebbb5610d52415e7c8cccc5510dfcf4cecfff5104b75c57c20f3d3b2503f1d78e2574003902d66683c906542a5b0243cc29ab6f62535db4478dfb2dd2f30cead0ea0a579d594242f8b1d612444f126c26d878f8c0081df90971be86d6a7f29c4e3de77f7130fd2126038c5816d5d90055f85375ad8874e02e485067a791d880dc0f41e05eae9309da935d9fccbabd6141c4f8c1e393c7fb7f7f94976515a1b9a0e3b141df3ffda079a8866843425c2929550dac4f59e1ec47a4a0b25b651f6613ba08e31aff45a365e5d77c8b7bd8b88bb10fc4ca0b2d236cf36ccfe9fc95fecfd62795091676bcdc39381ffa6b53c067b666264eb738b3848969b28c3c8384212d679d164b07af95f3863445ce4ec9408f1652182396b5dc3261aaae84f6eb6e552899b0c3d8906004683326b64397220dd556c91d30e43b5ef2045d23b99faa517bf8972d6eb4afa48da48543b37bc4e2dbfdb12f871492f202c206aff2a607cb9c7e0fc580123f540525a0acafd4962bf48e3705d6800c0732944a1e71d8ed2c884b512fa1e9ff0907397681dedaed90b71eedf9d8369f751dcfe49a1cd376461e8ec3f9589f0fd59ad202d8dcbc21018811993e41cd5c3b43054175b2e4bd6b09cf27197a26e381f7f8656537ce1113085dfc90e8b97629a3785fb6ed85f442b39640a0525c3094b55416fa73898a228816461db31fc370897f0070aa42e35b21e1984feed7177691236854c8e2d1bfad5055e0e02acf04a343a6fb1b2e93b9cf002b55ac1092c2ad8bc7d54ec001aaf4473f44c6f3cacb435612f3c6b683fdac02e32e11ce1e7d5587bb5abf2c8d3ff5400123eff17098460f563b6ad8b01df3593bc35bf12ead873db365676af907174eba5f174da46690cde0c16eba96edefc3a49be70cee7dfe019bfd48965ea9b914b9d4a1f36c10301b37193216687b86847b9d72a9dba2a119a18d707ef64b1116b9cce49db2c4b949efe4c641e9d3ab4da9b66d0e3728890247d68b68fa4fccaee33225857d7fe860d4df1e3772fb0e2856a35538bfcdeba4c1b12ac51a508bac14f583207f9add69051faa5a079419b555be96aeccceef73e0a8d8c12e1afd7b7735b63aecd119a3f20fdd13687f8d5dd20f32d0047497b8fefa71884eceac49d37085a04a662083d001855cb3180b8e156df84f3813af16ca0d09d455e7262f19c5ac02eca2042098fc686990fcab671d536be0084446658f45bf308f44de4b9e68f1a0b7d0e116e9dc0933b320e47b99072d422c3c2b5bff63c65ff62051221d06d43d44aab22b68f436efbd4eee342d0e96919f89f4a8931cf88bcd3d431ef59da0bf1ef68a3b63f675093c4c7c4c421d4ff54ea6b8bc31b3e69cc05df0d0fb378a54e1e90645db9e35590b59549463d3a338f81ab6a32f52ab7e97fa1c3b808198a388a19e624214462524274009d5d84d5beb6e827f9b04a5ae5d144128d6d84f0212ffad5099d97f16d5ea51907fbedd76daec03cf1182f517be0df13df722801b217a909f90e76596b142c5c7d35ee085f1eaba2fb8c0b1db480f6264141dd3fb5b18cc4033f23d9867304f2e988a064da34cfb92aff8884df44a15a9c5303bc2ed00f9ca55c9abe751a43d7719c1d1b1529ecb198754b3d4a003ec37f11d0c1c76d41efd36c64c13f215fd86e8fd02b1bc30e9d2eed8d6a88b859fdeb4274bdc60d99eb0f8f9a2e380032a83a9db84bf1befe38b4922076cc2cfc83d9690e3b5fcb27456f142b11093dd7bdb7f1d51219ae0527c3d7aa990755877a9071d65f69c0b04753a9f3b3b00aa381ff38adfb9b98cc573ed3984e7f6dc9e629c6f2ec57885f80d816cff0d8aa4027bf132448cda4a5d0e1cffae3e88d0900d1f2dc4fede24d77168c9218c80b87f1381f4b2d84c5c7f2a4729256a64d8baf4ac4de0bd395f47654b029563c6066b59ad42ea5bd80edee2a06a62b04c0930442063fd443ebed52662325e0d610870e9ecb571edcaa992afc6bba6b7059c8d3cc0ff4f22109cc36bd33a3a53efeb16c377bfb5c42b2d60218c67fa22b84ec7fb8d44dafb86fdd7da314d357a9772365aa1deb68c786d818ff73d342bff6e5d138819f4cf0563a90c698a56925b6da78c9d07cb4d4a3f4d6b78f1efac2659d387d49cbc69d803e1acdf9155218afdae393d50d6346383a34cd286b68f5e8bdc6d8b99a1d1d7d2702906e2aa9a9a04d7abd96d129c99be31656eea08e8c3347bef29e2b44dff01358f72c949ca56f680a90b73e3c5597f8390f87fa2858a037845ee7764f56f3310cc2796168ea43a7db267e4ee36bb8a11bd3c771337941deef7676ec15ff3314a29b4ad0a679ed4bf90264401a71fdfdeada3666604fb05958fd46be35c63f1729efbd18ee60a3173ef7a709f6345986b77dd9f9b719f512270399206834ada721bea6fa1ae82c44874680b18ff57000619c7f09213e7e52bf33be4150b09cde207aaa6b47a772e077ef104e684561dabf41f74fcc864cd4e2296aa08297cd8fb99918a7b2ae59d3798d5ef19a3a98f416ecd01f42a58f8ec4d17c794cbc635ac38043159871d2699bff14697512763e9c692f6f45f07067dda04d494b47cc0f7fc62351726f83a5ea99a37505a7947d5a3306279f99599dd66255a06dd6f49728584ca036cf1952a9b469ea4549e762c077b6ff0a25a3b6dc5fe4cc67b254a1552e80471a4d2a38e992cc09a9deb211907a1d68b118c6f1430c18982810b080560e2c6e76c17e010b2b6c7f3e329707c3bfb2cc83249923483e856bd880c6133d23cfd7e5c2beb1a15a2d88ecb39539b645572c1dc9db4e2579c8a9caeb96f03ff91ed41ae668d6c8c16bf6c9167f8cdd3c0e9a5215965f6cfdbddfe7cd00b4ca702155878f9aa9141b72ee3305a48b8af6b1da38e639906c35de849233fa171c021bd15c4aa0df67ed2a519792c9e420d3a543150747db0b19b2935df78a47aff348d3796b9261baa850a8aafb7a215bb578a17939f44eb89f1cbd9dbbfcdfbd0a519a471578f7582408768d878bcc83d31f7b596bff1cabf4fe2bca47cdba816a1f3ec155686487bebc37c93a3944a497214cf3caf4b910b1a7452cc6781972422b73abca8e0571e7c5ade4effb63daf60e5f44dc1837f24ab5e1af86d025b6ffb5f6236cb7fd02520ce834d7f31d39e4cbf7d7d17b583941bf6a0bb5099d7033bba1fa6f5cf0ef47b4a5ebc2e8f66922d76a9be0bf2faa2ac172707f8a19bfeeadbe4f4c811ae8ce86bc636028f55a38013c4e39315be4a9e260cedb043122a1a4da3c6f50676ded55216fabd0dbaa659a6ac47e48e784bc2e4f7b3148739d6384bd525da874ec4c4c2d8d4e62b60be4f47f20f40eadaf54accbb106acbca466701e84db4be82be84c8c5736dace8a190519d9bdd5b71604934371cea43eafc76d833e3ce10819d882b4ee33718e74bdf95a0fc040c519ae5f2f47df4dde3996bedf64e401f85a3a01d3f51bef6c076181069e40af6c06c3a16c444322537e400a1b0ec9abcf65a5e7b39938e61e05ebbf79b6e6c100d5a30f23d446fcef89a4037cf459fcbe81e1d4e73cec2c32ef08e173d98766d10dedbae881f6d4efa8ec3fb8920ca77082aebe94851fc518ad791ae27d547fc6709b6e659a0e15e48914dd7f5950533db954c90db131b0d5ce3811fb3b5b6d41670fe450b2bcf99cde54412ad3d77c1585ec32c85d72a47817c63dcbd1ecf2c0f6bee6ec67bf82bf01a63da499df995eef91703c55bc950bf34369c929da4a5a6f4d4b38484b2ed9600927570c9fb4fc19da7f3b72c0a6479685220dbc332dd0cb37d9c9e8d8ad3df586efe65b9cacaf0aee6f27b045e40c22a305964365f4096f4cb28014560aaf0228edc507a62f59c815e17a774b63d57836c1d4b55cadc2d267a3419805372e3ab917f1ee1e8176d0d730361cb62cd89d8d6024d8fb007dccd58c6b2a15747634c705bb1f0701155c6ba5ffaac18954f74abfd27bd157d60e2e403ecd7a9a92da9c536dfb06f7a722441b5f6feb97a770da168947a88c7aad3cfa9f4c18727e6090646d4f9b18fb0534e0d5aa626ab73c5f996a15704503a9c5c6a2d28b683d49f777957120b73a99ebf040a20c1ff04a3dac545595a7f2d14e3f3d5cbd432dc6c1a8495ac16f94bce458caa2aae08cc44118be7275ee894403eb074030aba8f4e886d7011bc74b03596196906c6e0a8c3ef73c2ed6d92e71598a99e2b4b2b7e60ca462933e3decdc9461535680f65071c17d329ae91c6bef8471b12af776d1d273c5f3d39c9b23855389a643a8a9f0127f6b581e15fb5aa29fe8d60b75a2a65e219c003ae60c614c8c488693d5e95f89a5884c6b881efceb8bf666f177b5df88c606fd45ab193e0b273850b7a52daab18a692c1d016877fece51b9d73abda8471672c6fd603de198d8f9e7185789272ef7225a196a025a21a3210ec775b2ea3068840a34eab6f7d35c8b0e25c8eec6da99957c0e25b171b8e518471ef63fb0e96f22d1ed66efee3ea7f1911f2cd4384d01cb57f538030b640486cfdce7f9bf0881c8cb11055ee05eb2e9fc2d7b167e7c2fda3c80533a3e0b353eec3cc5cd90ac744606f9b2009573d00b73e11f083ed00f3a6a4aeaa456abf6085da1f95ceb6b3dc84f2660038e728c9c4b94d1b57f5194d0bade8a33367619ae8bffe710b3d422b578320d5115192a922ddd0a1f48d63477014b5bebfc948b2a2b59ee893524e060a7e816ceb664f5cdf7759cbe95714072f357e0079af966e402eebfb1a8e1c090f9ad2ab0f883c3083205f597f72d723ca8b9c30833b373e27052eb2e08377cdcb49350aa13b16b35dbb7194eae6501c9b2dd71ad873a51292a55eeab15d9850dae731aafd19cbe7837be83633fa312d7bf0e7c212b774e6be6b664b0777f01d557cffa41c23ea7737b552994096dd6f06843aceeb3c3a17f637cd5865641d90c0c6d1d5b439689a19f5182885496f553abb9b4123a956ca0dcddf5257242246636c8ea73d0cfe5db1d9fc759e723d0071cbda5b6ac483ea5f156a365b5b82e0a3273ee4293e1c230a29b5d06f9ae83f4429492ac5b764be6f209ba17d0ae0d6de2b1cb9b72bfde885ada2a91818ebba779af1d4ee4a4edfa0399c5d6c418bd702f1ef0588d5bb3653b3d7eb8786cc90b8d6ef6ab9b7a4898dc6681d4f000708a558ef157149e3fcf5f197d6ccf2599a97dbc066103f491ed424bd11c684b4e9ff262e4096a252785490536b676da12617ce12107f64a8b988c3c8d65ece1ea921d5f8c655b91e78f8e2f9f19969f31f5b806cb9898ee53ee4e9976f2b2a6730f8a77fcc4f4eeeb4f780de1019b2ea5862f7d53e35a06b45466cb8b40b68cbee1087b35415f023f86956454b053213e368565befceb5f0745fc8178dba8933ff8862c35f286447c8147a3d63ca4bbecfc873a2b49788090579b66c2ee25cbc4208ac0c1313188b37ff9a3f5c53103a79d8b3b92008044f6769dd45b4f05909d096638e38794f3a1b54b4aefd381e1b9173366065731c581c44fb970cb16f42ce1d6c0612d1e88fb3b444c58e25559de7f632ef95a9baebcc5e7ea0d7ff013ab5ac55789749eb9f250845f9f258115fb3ffe9b22cd13fcca1bd837cab33081fb3f2866ed2b0cd3bf3777054d1f61f7d070894eda7f59abaf7f0a01cb69bd765eb4a153488bf3e6c5b0f4e47dcedfb40affbac56ee8291c5cf339d8bce560312bda813aa4b48add4c9235b42541a7a723643a0d7ad7a1b680ad589c89b77571d40ebb8c05f6250355bb169e0bf3ce4a64689401be7ff53796e415c1c3264c7c85fc13aa40f35077d95d05098d9c962061dd6a2afab39f5145fe69a0923b82da0e04615639d74a2235715a7e51cae966bf4ecc94d27293cdc1eb9b8b49488a436a24ce0d8523741b52bc71799b5264612dfc0150fad48a7699d95ea382a9b0a61855c9e0ea688000a325ceae92dc744f01c034fd347ce1087a7bc518926810302b0d186babae019d8e1693a2a097ed1a291646e1a85c03c94c932eceb06085a2930c71b9063637650bf7e09ad7fbb5f5e7e429a273fd4f1a7c3d288a8d53330f2d3f0f7c9100f36cf2aa72a68226cc758f05c5643643ba33df59077a63556861bf1f491ac58c481f430a0d7984db2de3726ead1e97476476321bae08c0d99ab6bbafe7e31203f675b779a6007fbd98453715c6b64317e2d8d01e23e25afa164bae43a732508780f38236ce25e8a31121e650a93e9205f832437343901926d5121c633ff078388f21a9e59df4ac67aff27a660cf2fd407bdffcaddbb90a78652e5419069e104632299cb04bc8b9fdc1277cde95e70a9fdcedab5f41042b79e21f8362de123a813f5891c48d7281e2f7808db16c05a47c9fc4cd170f094e6502b9405aebaed273ac1ad5a5936fbb9796c9f6901846d1962da6e1a4c83ac5773ee2e6974fb97149467c644a50c7253320a9fcc3b790ce135cf130d6a9577f7b8749eb7ed892018b3b899e4be66a56fef8062c79cd169738589c744de81550381b3d6744155a793552a8d66cdf364fbc76eb55f47228a81052b7a7762f5727053196440a93ffdce13dfaed06e4977fa17817758d527d9de3baac9023c7cd1d33177868445ad9c5c809bd1fe64a2cb7754e0fc83731f3dfdc2b86ea248ae5da1a89bead60f64147a7062fdf6b5decd6234d19fabb23915a4e6b473b1fee95b68e5f74a1d3830cd3875859b4fa755a4fd9119f16dcec0f381e676cfd5a26b35c82ddcd300acd05a169b703e6cce39548de36c9eb26e1c888944164957387d0fd5fd4205f20785663858b8072b1e41044bbf84838829d5867a00490b9e547e3b7562f2cb1b866e1f032a09110166e31c522ab21eab4357012e74938c4b5c21a1a4dfd32f8c8dba7c7259087b8f19d71f94fe0476176f364d8bfa135c737253454ff96e38b628776b8fc10e05badcdf731d23e948bad72df75c3cbf14c38d745a25cc81e635e3dd46da05226a263c3000beea6dc6430085b622e90a59fc19835c3b74ecc26273620756684ceb37c2204051170adc21d0485c3f5e2356ae7562f02611fa57ebdeb0b632b96d0ad2ec503fd1c06a227b3705fb906dbc059ae84814dd61c77d5a0e0b96e3f48a0e1ad7f1f2ed564d50f683dc89bbedf99ec0335d51cee3806b12ece600e13bdadd445a2bf7f28db90052a523c28feac941ee550bfc6246a5a85dc2c7bad637e0f729b37bc73471dbae1b643e5d87e259131020605ab502b9ef0c17cbb657c53f069823efc4217fffb3d691e8873299d8266721d51c758b4078598c9c1d11d2685370d155ec5ef84b6a5d201675861ceaf658d4109f92a9e72fd83253082e1a5380600c8c36ea35457d8785e1185083dac9dcf8db9597d3a08d8dbb2611ff62d6991a8bef98e666ffcb46266fbcb8ee19f69dc369cb0b03abf5dc43e395391d7e52041bc3bb0495b3121c3a4a76f03c35d48ddbd7305bfba80e75a765d43ef16da9cc8faf3ef743c794466c52fb90ac730b8b2ada23faa3a55eded6793231f7b2518c69a37387b84b19ffa982c705eac7d6ecc0e57b017291ef6990e3a2f05fe09a314ece9b501d4f222140a2faa25261ef1aed3f31bc51c3673af200c1b3c8c858f0855d9bdfe9ea95e4443525991bbd50dc52460d0c10f8562bc2cd5fe0308f745b0bceb6fcabbc1081a1d14e935c03dfc9817e5dc5107abe901bb975f07b3818d495a1b38ddb4c47aa0a9dff86ba18d34044e6594daf49d792fff17da9f721084f8dbfce2614317fc7834ae807ecab516e7da751489558b3a5c649380854db75b657d5196e430fa64d6259875ead90cebe3120ed3a98447a45556a73ebc2f508e3b3b56cb2e5eb5fceefc1dc84ad1867567698266b42de80f67fe1f0f82d40e2d48d4779b26594cc8048cca2a209c88718a33555c38f26892455a3c808b3e3a6ad49ccfa23a609460db1f7928fe6f330626a662cdeddb0a14b643600f0db1f67d88d604ba544dee6c9ec18f23c066a0ecd51e1a78d167d3ec360effa37c661b40d911e13ab1454541851d0047a4e6074d3f81b87d3449b413d86e323c01aa5ec67683733f05e421dee3f6ee0692115f74f21094860d2b3f6ee9eb7a74014e852bacad1ed98fde6834a1230b8f16ce28c0232ff1df788f91ca8f0bdc53307b8a46eb4022c038f410214d09c68888081a6f08431827e03a2a5456ab6361098df0ef106c9112f18c917f97836725d01388d06e0637cba404ebdcfb45320a2ab10499b4c8d512882fc5caa5ce0d15982edaa1d03fb872e8b61b833d3f02bbdb3c894bfbd7a9910ce11eb95da485a88dafe60076c15458aa25ef3fda187a1364860dd207c0709767b344c8895d06b66f88a37db7dff8d42d6bc2acf1ea5f1ce6f9e9a273f0c84e0e8dcc9cb0f4af2d1968ef88961b80865bda7db91ec44d7797e6e62be3c77f6f5945ca04eb9f0154d016d5e4bdeb2b0f9c2f814920e10c7588eab1c859117ca0718be186cdaf7b7c2e949fe4818300a188f59e45d77a51efb0a9d89a9f99a5bf7e40d591d85b4b65298e4f223f00e9e4d6045d0e3a10c620859bdb4d5500344be5d2e9fbde2f01bd899b2c5932a85093aebd734eb20ca16f6efcd0ef26e453a3684db3643eb1631061dcca69cf525f117300135f0a4820651322c6ab72465f4fb114a50e990e3743e71df915c9b4435cdc4bc1d63ffc942132cd1cf8cc80aae9099e609dbb18f9e1c60c178e864a5bea1040b4e95fee935285ac7930c52a71c427278eca43f2120bb0fe2ee7972aacdc6d3002c8aee37aae7b8379c290b2fe4fb1c3eb7b9494c4dc55cdfbf4b77d217ac6d72c00acb01822c22fd8ac972fff63fef88edc638f8de8185beb7edb10afec1f344c4a46ab59408fd1bf46da0cadf7fab69d70da53b08d883312027d867dc708a2d60a9dd210f2ac53ea5288bbdd33737a9d1f34bc8f48aedad9f5792bda6dbe2da86c2ad54b0c78e0e9d1a97b612f4067a623005089352501cc4e6237fb5352c418d5c36f5ea1299df8053610ff37c6de68262744323d77e31f1551b86c04178c96d4e662d6ec3421a09661033ad4462a23550cee84e70e3019278d81410ff0098ee5604bfc32afdb688da229baac6fda3ae620b27602af50f1b34d0b41f71c1f441a5b6ec067503b0a8ae2f0e33d0b45d647252552cedeb9888ddab3c4d45c1a4a021afe1ec1c4ab0ff0fb5dbba385a9415e8cb5fc9927bfdebd93de14c4971e5e88dd493736125cf657e859c0a3df6c138e91014ddf556dae5333797a9b189c6f9a447c93f7b2d99504a3fcffb84a5fa577e2b13e678f08f072f3eaace5499e246b4c823efea98ec52517072870dcd3f35512dfdd37981fbba8e7429bad74a918eb179a8c4f8757423919371e7cd978eaf1ca264c2b4cecf629bb6c9e5d7b498e3808029e002a1cc4cd8001d69ee2598643b25be14a898121927afa0d239e4321dae55b4d9b1ba53d2b90b44e1710db107b74cb314d34ef33789727a2026d04b14bb017a23cb7c8300fee87d5a967a5a00644c1d52fa17c4c3cd605cfe96f01ba40ffa7a52d4885a97929e2d7298fac9cde983af32322b6afc19755d04932a024a45622b671980109fe3034ae9b815aecc5e328d6d57c5c179553924329b21243b4196a241b7877ed86b388f7a8906f242f9bc8e9819ef75c712f48089701554031351ad55a5c2628b398fefe40c8003ad728c5bb8f6f64e667843bc3b928e4ee76bcfdffc31e94fe59b7cffd1e386b3001fe6471d8841278e912ba4e0d7c32aa4ef338977963f2c14459fa23715871c55f2ac2a22cc9e7099cbc29aca43d314104f7463aa006a8a27ed229ea90e213b32c95c0b78040a41dac51b66bfa9a4989ed77b0b2114ceb6df08c176aa04169a011a469fbcd5b6792c3d43aff49cb2e0606f6db136c09d87674e028fd37d56ff9ac55e0f31efa4ccfd93bc0d5341db8a2d1ce5ebd2628b2c676c991321b621ee28726c18bae3a2a5a73a09a5af38b680c37ad71b051d0cd867643e83975670c92676ffd6277fb4d0c33403d06f7f21de22d76a84fe3b607a0250b1f22127ded8756b265f788cbda44e554b002eeca5d3b0622e78f284ce86c76f47f108154b7457baa011f3a3b340a1d0ca277f853f32ad050a78c0ffeb39ecfbd26630c5ec0b039ef7ac5deb63183fea1341b6b6cc0c160bc0420a0dd65035515539c3ec787d324a66ef80a5a9b8c819b7a2bb9ea3a98d14be83dc87cbac5b41ec7d280140a13c89ba6099db301d1e0f584d8c4f29a1f59412421c1f82f44083807c209ea44a68f3e1358b1336857ef1bcb1fc09b1e0fcde3381ad1eda3a5ede702e4b7d51f14b25ad88de858a2e6449eb415a846b0cfb8d0ae2cc177700f81a83730e6e084df7352969e64b2fa410a32993f58903c0990c39b8df17d3785224d468ddf6bbc3d13aceb3d6554d72cae7bd4a5b02cff0201ccfac1b426a4644f1b146f5f8661ceb22713c3cad94725c8a7d02acd5f40e66213fa7563ddef29bee2a99e470b96b0e650ee8ee27b445694e1d1a05e53659b8bcfffdefa9909183cc92a36c9efb86bfcb277596fc50ada7f413c0ce5e7ee648ed923260492b19885bf4eb787d7ecd3510f15c10ffe6cd7d14257628dd90fe41931dd3a8a9f307409896e1b8d876e615602c6d023897706c90a4cfb9ed76c1ddff939fa196fca7ad8d82d9ae8ef71a4c67476435af3d06891c10aafa748b72f80a4ce84a549fbdf8b219517acfd851c62aee3268f03d796274c7c0d74c1d95d3ed880ea35d3a76e1dddce2c87348786649ad518f12f00a5db5263a70d324999be93f6caf941e37fbac08edd3d88aeaabf1508781aeff4cae31b23ced8065a4e86e5ee1252b007ff884e63c338111b5a1131ad69a7a0ce8211a6c31f4cdd744b14f1cf824f78ad4e978b4f43b757ed48310d9d2b88b8944f595b065355468aa0b3ccb058d43c26ee8162893810db55f7683d2ff841d20aab2f49e1640d939638299d7389df6ea2d5ad994861c83b7f82160e0683bfed45d8a9c8dffa5e9b55a418b0952c4b0c582f76a57c5c5674214782d1823ea0470f9a399d7dde6a14d66e44010e3538ef8ec7bb31b06ffd4226b278b710518f1fedab126e5521f30f6a2d0b62e89dadee66dee532d8d41fd905b91468d75cb51fda2b0199e799db357aaf10247f8949e62ca965f1d9600a9f8a76fb66a4ec1e45d4dfb93675f1a87f9d22fa14c642ed19ece41335b220f2ccc9c88deb21eaf05690491db36fb61cb5953ee8dd1dff07269208a98a230dda82e88733fab9509be6639ff9a8a3460bb9555157373a9eb313e007da88181a9745ab291b8a25f0aa2cb7ee40efb4e5eb802df9b4e24509c75be717a10b45fa4404c6fe7b31d23aaebbbda44dcc2cc3928b620aabcc4f771dc7947b59daf5e49fb5b13bee641f639ce0ebbb3286363aa12a41540c225b3f081fda83f1eb5bb065a53b2a8f6c105bafe280bef4d84f73f60da4d1291e925731a4bfd50026893d67839c20f25b7a3f64f52975be7c25a1877d77478b8b8134c57c6d2e043c6555bb6927142b972c3c70a3a7807f83c5f110fcba676248731ac51078580b1a2f605adb892e9b419f1d9509eb4fb9c5f3fd92e47dae7cb3596e6d04c182834e6ba1d37f139b4f89c95113a77c2c1f802b6ce1624cefca77918047582b694ee5d7da62d97efe090a09b6a3e6723cf09714adfa91bd7dce2f6ec35d53db92d7bc779d260372518d57c973cd07f890c6d9ca234116560c1773a7320aea412b04341c29b32254caed021817a12e9dc00740e76b13aad96dd43544ae873175c05982288134aa8460002515ba8779497ffad9699ad248c11556ec8228e2668845b3e64a2d202b6ca76fc89eaacaa6568ea1b9f0aa6b80ba0876ea58441feba47fc37a0d5a4a0c313bcd5a49de1317eb0498d3f17104c46e0674280be32f5c691b5fd9748e37fc0a00a12022559a411402365c37f74cdf90311cdd4d810929f2f8cfedd690ac1fc52e0d51303e43b070fcf8314537ec4a7d97aa19944a5496e919c72b9bbf0addea5855dc99893e0e6236960ee7cc9d496ddbbb040c9773739ae79d9a5cc95aa119a3210d5e18c7f3d9953050929fe2adedc7df10085c057f95473c6bde947df47a4d17c9a934b95aedf0bff303051428610cd57ed321dea4b0ce12684c0803ad5efacebb7a826a6be3732242d67839f86f99a17dcfe5befd3f903";
        String expectedSig3 = "00000002e562f6fc4309848d94edc37e66c4fa704a494576668583b3325e859eb7c67c9f74ad76223a4a61b8f9d6bf9f9d64b74b6a8f41f64ceb5c9166787bec71f1c55119aa1025f6d9136555740a82467b8618bec24f91cc75ecb4d6dd714908847debf0185505d51b9e64c8dc9276186e7dfc6038ea0c686404401d17b5670a50002722097181934ec58ec3d871bb206f4009857a2bfb810d4005b9dd5b32da5a0df79c8d895a97e163947fc96792b806041dbe480a83b0f6c015967961173ccff05eaa59284736ef2309792c04acf3eed3b9840c45cc03ebc337e9a3c7a853f23f9488c91f27fe7440fba480e6955fc32ed4002aed645a61713b0569bbb40e17b648d67b539fc4ae94498ed6e7a8773492c6cf1d76820fde249b3574b50d9ff1f52e36f87389ee9f821d0a41bd3d224a9d12f0bf9fa54692b477c1b4fe7c88ad0191084cf0bde10db91ed8d3e639503b5a919640f63ce9c7008030a10759ad2f029a54d37c8505af0eb90e8e712e7d0276ddc439873d2c73bd75dddb936c6ab4794858d649d5bb291fe26c43bcec71f3aa11a7eb40cb8d103978cc88b9ef2a26092adaf6568ca4716bae7b7c6225feb2bd53b54685e51fe4158f534b1018d99748a2bdd8ce75361633a8a4d3cbf4c46b0b9352d89fe97797cbbddf6fbb211d845d47e4bae5a9bd752af1a055acf9dc3599f0edb292412bbb327993abb33ccee77af9fad2430d03f9d23dbace7d0081b6646ca141b3c1048967874e9f954c1d5ca23474deaf7188469983a7f80ddbcfa107edac50f032d735f38688e890dfa8944d676147a0f17e069ee4e0ed6c51e5c2311bb72786d4040d0120ee7b303f404625e6161994a34ee9a407fee7011ca90e97b54d0d458c397b8fb2c4dd4f54232bde55ff6ae23e7bce71a7096292de6f0a52993d534d513deb7309227c43cf07804af2f3eccb63e0691ec4dc38c8ed53f58822bb26101bc90437aa1c61538f48a73c4f6d0732e9f8ccee475c3d1bc28ec89328d4628a67ef167e721e09d54fb1ac4cd61bb49c0628cd99b56f8b70283f33f6148b0d14e2cb95774b3663b67261539e2b592d248289ac56ab3536fd34a43b3990dedc0052d01c4213df5995b35b0527d895ecb9a7f36da2ee3ade2181debeff39923f86982e3dae1d67edfe27facfdb8ef36ae5e14d417276a549ce352bc22911147ec791cbddf2a3f4191bf79b36221da26d7114e3c167f6d872a80ee39e5820adc14e41e25cef9300df3b93624a183c57138dc1627dbcca3e250aeb9514288d7bea14bba07f53983967cbbac39046a2b00c85b2764a2c87d52c73c1d7b267055ccd2f382f8f39d3016fb9580009a780112e53f748eda91262684512799c8648ab083dedee56fab62c5dc48172af7098e774ac7e369a6149fdc5349eb62677255f243e2c022e321b19a051d1980a35fff2d0d9cfa43afb01455b4802431abba05b6c47a73358abbb0fe2da5a5d7b70866b83dddd42148d614c049fee0d390cdf742a4f76df41a990d2013eb1c6398f4193f1409eb1ca61d3156200c01a90090e27922dc23927d621aa9cd8aa3c7368fdde0d465fd070c3090e955011fbc010d610701b12fbb230296d5b2894e18e538f566071cd5bc6c85915415be70ad983bc57081434a6d470f8322ee2a357be94d405389289aa406dcc4d0fbf58d04b4347138bda515b309fcc16c0e0428dfc9a61e821690bb059465759240c86c6633e75107f7757d89deb7d6d1774f066e37adf136ef1bd4e243299d2806e9b0104b7dff8d2ec617620170c139264697c26f43370cddf3e7d70e255f70e99db56aba6021ee7cb23ca3dfe0eafc659ba6f3318241485ce190316ce163b1e44cfaedbb9dee53e69418b070b8a910e195aaa8a5c627a1a971843aa970e5d3daa41bf628a5acb932412de7a0f757fefdb4b380650a9336a7313aacd7de945883b3807c2f96e741ffc3d04b8377d9715386084028bb4f598518ce31c64455c58dc9f8b8694109e940b9b78bfdd45dbff61d5bb17713d6774078a853241b43dce88f4dd51c07d43037864b05761f045fb3d1f45d80784334f3895beaa40c43d4edb3bf14229d438f8944276916c7b99312ad58a5e295adef49c4ae1582f4077e5e42759788dfd1bb63e1425725991c8b6bbf22597353e65908bc76be7b4f737d716d294349b4d17aad1a4c523b67fdf7c54a627d0e9d459fb9de783c1012d72ca90f58e18c84c8472bac52ea378a44105018f501ec476dd53632928df294fc7eb83d06b381295cfb2e47071d9b6f1a9bc440999771b9f2a89d06b36647dc4f787696c3f09b216f2a2f37567e39fdaa88c44fd2f1ba92969798d6812f2d43cfc7b15efaa706cd8995368b1d73b86551cca835aa6808f56fe42e34df175d653a619a3825ac15f1bab5e4139f73cd555a7319c8221dd9488b0607e9c73320968872c1ef68117f083fc54d4dc4ad87f4d3e1016db6d70dfdc427df4ec80a04ef264e5273615ba83932cbf0805f6bb31fb0d8293912fac7644cb4d8d17d38809a6bcaf27e0c06ca90d5ae6a28d2bb8e9026ddbd8699f43bef324e1bbf252d1e1d476ad4bdc5ee8966149f93471dae07dd2f46e668afa94838c63c432340e0d34fbe77434c945b620a23fb7559758def29ba646afc7b47f9ac58446afe47125e4a1b21cb9cd0e2a296cdfcc23a2bb113605c414d072ffe32cc74b9a6adb1f1045799d657dcf300ce40dd040bdfca91188d31a847d4a684da65eb9bb4ff9cd60dd99581df275e2267af6b9ff913506e39e563f63cacb06fda15c072d4192966ea623305ce759a72fa2f479cc5e46826c297f531e79b8da89b10ef955043aef04976338a9719cb650ba8ca4410f9436e975fe88adc1e66d6c9d2b60d378df2df934cbea13a5ccff890006279477d25e10e11ef86cabf240535e79566e5251bbcd824aea9a39dcbf450c18cebb3f4bb5a1561b1a1ce792074e6b6fa80ac3662a5b921027ca08f8ac48f0607234b7bdb707e99075e1058e2e168f6e4504d12b9d8ab956335429efadcd2db1582c5de2ab1c9a245f87efd381715e7d290ea984c9502b6d25ad570849a831dfca6d891391da1ca4130f05c610dbfc676e55f3d76f9edaedcdfe6ac6d14b7999234af814b6693a15eaf12d4684c767abef5c41c632059b04225c8484c52f856bc2d92008a924004b4606b097d39ecdff3c15d2135f0fbe30d2e81ad2e1369c282752772760928ed83820b6dda6deff91bb6349784fe45393832ed7a63e9da261525e5d53f2912f94e5a113abd635d69f61085621840bc65e252c51e29b722e4ec9211b739ccaf3d1f529ce90ababb3ade832f25b938d246087234423c06d2b67931c6b3ca9046438a54343d6c2c27d8f1b50e00100373ab18fe012b977c210df35bf469f9a0d1d6479361b5d0c0bcc9b2adb4d1795e3aa8b9bee11221f3ab5b3b09e5e021d991e1eb3e783302852493a15b7c1253cb22e1c09fa6bc8519810a00f5305798b3c2f539d8bac5d268eb3527a8b0411f4c676740540fb0a24915c53f05b4a37a9f30024756d0288e144847a2651ff8bcb40d0c2c189ba914fb25d4f413d4d0ef0bcd2c261639037cdea16394e0c42163b61366e4bcf554c291c252b10b3570e32edbc17ba223b79f323da831c83e37961cbe73cc67072bbbe1ccb009bd0c2ae4547ba9587ad05abcf479f74acbb5c7f78150714e356ec88de5176084506126f5b21b861664cedede54eeef7ae78f8a3a61dbab5438163d0190655044c9a5fc848a0ed70b04c1ef62492aa1763e981b30fd96bc84db050aa3aae1c913a2588544e45fa284424835500b183a4e6e61649f0b5fc1ed255135bdc204b2fbdfd35d1bb7e7f243b113d59c165e10f385569d3727f410b5ee0a25f25d863ea28e04d10cd67fb6b2e4792c4ff3ac8442c0f4c48c8a012d5caed2f4291f6b3f8aa94f27f075ae7118170e61b6e5eece5c088278d06d37e2171e2809f64d1a4762cec78a42e427373f63ee54f93557ddd46d093e4af86223800cc640b872ab0d4f8dcd1d86fa5805db002a86f45c50089dc292081840fc5d7f3e5bd82754452849f777d0ff6baded3ac1e8198471b241aeb2b5477aac62f2f5aa39a30580c2d233324d464499d0666793ee80ad0a03356fe077b1d23a2185b9dd122a62860b5c639de77f5d7b502c6db66a1c110dd596095ebca95c2fa003585ff1b1424732f32b1e209eab65049b6909cf5ca81e2556d0b80f8d227ad734e6a9dc5fbeb5fbcbf77cba4139d9f09e4e939280807b22b947f8edec895f1dc7b770f1c341aebffd4e1e4866e51bdb4867b30f600c6af40b86ccb2368eb8a769a879ecdd3d3bd52fcbd8503ec0f30cd4fd0cb2b51b9af3f80023d5459ab26b485a486e3c26de41898c86525dddb8917e58c374d395e85b299b5313122e95a23ed734a2b3ba6575cd8de8d61e6313c3066ccaac5d38271e93a39583f1fa01bda4a0adc14e6030fdedb5b1ba2bf071a8351dc83e9e492d68804bc4ae8d66ed93f5f8f1ead4d13394c155006f3ebba5f184b3aee9f9a0f5da94c1633ba2eafd679b8466056e41fe844f7f86e37f5bac0b69abcb4e9fd9e04f35eca9bcb68e50f28db177b84df8abb52ed988fb06aea6b95120c9a8f3dabb86ef40c99b1de25058071d0624409d540bc557b2b494ddf39ca018af1898eab2bae7693cf5542e7340e355b974c9e6309d681247cca5f5865225a21368cf41be4b0910e0df5be4f2aeea980d3b46b30bc644e6ac3e2eaaf7f3176ed9fd54b3a9d8729204006e266b35c3db40a56f999ca3a03fd6447ccbedd0ae1096c8b6f9ce462c024d22afdcaf0a3f38f8433dfbc494e98316761579e164d730b3a4cc552270d95ee97c38e02bb072cc16b88058e45ed25aa21fcef5ef55dd44406d712981cc3c670f283f80922a787ac8e1c9237700a3c38daf7a9387b4b778b96147daf7296c9929d188a205b73b8891d1a0c58b2bca842eda3064da08abf2d4e6364c1e7bcf3e35fff0ec195fe729725de5db371b033a8d5fa9a707f722477defc729bfd8c324fb9fb9331b91ebbf1ccc99b509c5c554be57ae6ad24cb5e60eb96a40e9790b8065b33168ba10514c08374d5c7b199b6dd9f0615a7914d59f19e9252648d60ad3f423e963f57fa194d7d5568c94a6a99c2a26ea037ded7e89d98568751568d0feed980ffb6d502b139daf32172af2deeb5a25928d20b12ebf4dc041094e870a9b000a94d6d63c9ab93d77cc9f1cf36b5d5f180c4a7e76bb394b78db19fc3e1d3719bbbcd8a5e377394cd7fe10013f754d5ed862a79d1fe1ec5458d5bb10e9f5310547084b2abc4d5d4f4850f268eac72dc8e3e105af4f63550b9b4d2838f0458f6b9b19da852ecf3c4de6c7b6411f7494e447af1899ef0e414938b152951553935ae512c84af5910af4892e8acdd08fe7a77a58cb601a4156fb583bda1eb00fcb1a6a95cd60f13f6248dfd46232bfb6078949b1ea583e8bf5909f57b7f9b4cdf3bf5dd7ddf1e4dba7b7aa6130da4eca58b0f9a07eebcb4019055caded4e32616b13c6b041d459bfb4c7670abf8100c5e30dd001db88f017f7fde72ed9da6928f0865b49ae73d63f6adf88d4ca7258ce4096b8221be737cfb816302a292209d4d7dd885785f51f6bdacc3ed4f9ced9a760560ed341732cd0b78ca888962d637c5d1f01682898af9ac1e56cbd4ada7c3255b3af91e5ea6b0a22f314595d85d040d0b0cfaada746620367516d410ac8e798ac07a34de6a8b6aeb219e1ecfc01e5ddf1b54cf7d32e43ffbfc34da07a6e1abefdd3dcbe814b3b3b885a5378979176d9c0968f753201746764465389ffe8aafe3a338853f6b1da6545efa803d60ecd2ad310b043e62bcb3c58bc2ac7741cc02c68f4c84e1912a67806f82392fd4264f691aec7b324324ce088e6640a894a9e5ee6e8c0a01260eab1b42bd20e004230a1182da42d6ddd57c0c3fadf31d913509e360e1684de3629f39ba8f9eaf8cf4f4ae50a60cd7585c4f5386a46aae959e2161765a65b3d8faf2abe7fb647d25f5b00a1454663e17899a92f382817787879f4d2a219d2fef19100bfef25b735de8bf8d00ce5449551a91ad0bffb6345dccc0705f9fc549ed47a1955c4ddc9aa06e4e51a165ed0ce02f771fa872a6cd82c49b8084a21e6e3b1d0ac6ca46f79b24ce72522965da32ef96f3e7a17f87282abe14c993e097d9988f016db35bda1120c0cc5317206fb659ed47e4adb17afd03a7b0572bef4d10abb0d2c9c8eff110548fd87c6ce1e46506996c0267b2af1c008c20e85c1a4f59e459ace14ac428572b7d9af57900bc5145be26ebe89f17c737c54f0a353bea5c2b5f5175793f9a34dc7b22b84760f36476be33e5a3bd5db623444c97f0256a84d4524941bf5446fd6f946cec82e8d4e47e2c48032a215dc26a664d5b7b800b6da6d6322560f7fa3ab57058ceaadcfc50e91d1581d6a8aef99325967c3f32e58782dac661c6d1c4402d654007f5bae4f41aa1accc50e378a563ea03b8892187bce7ab606e493b0e2c526398cd290d04605f351d4b927dc7f34d3a5ca8d61946d753421c58aa145931d321ff225d26c5abac68755e08af31cc40476eb4d4de8f4e225de3c48093083916d04bc6890c1bcfd07b47fc0001adb1c62c382a96da673bd7e34aa885bda8c91c19cec4f517475f58b2545c8d34176b64b499cd1daa90d2c9e63327ad5343a79e0b1ce052d72bb6c9382cfde957417a90cecc650366cd2f18217cedcce338287c48d2bcfeed7500a618a1b493aad76a442aed3be226ec7db2ba75e384b6c07799699f7ab2f6e86aa5b0bd24bc87719e700aaf66c4962a7856638e5a15a321dbfbcb0c8c38f7058f0baffe0b99fefa97e5086c3389d166753c0be24429b13786aed835bdc227e3c03e7a7c00bd34c526435ff3d6b6b4804e3f83b888a23e364fdef47ae8e0b736a77d95b1f015666d84cb79bd8b42e4cd5b37efca008e883d9cc0650f8132deec4dba392d6966390510d2fd509a1674406b4d90c0787268840e74507478dea704e8b41b7f78b163d649a27c59c75acb515ef41a8ce0f615c7ab05ea6c583b8f6aa82e837a89872d487e21bdb73dbd242628b331810fbced954e44ca161b0082ec013b019cf90782676d9b8e5b5ec63f922f1fb489a57d593e4f93afc0fa725aaa06feec21b3d81e38f57a30fe473a90efd043957dafa3f121a9f981810689bb4ec24376ae897e8f82654043ce6fa2f908ba69aa663b80b192cfc7822f1f06b655246b764593f8ac8c0d8f8bf7083298ca600a32f6cdcac230db8adf388e241b5fa008a77121ac49561429c94f2bda4177755df4d2a480cb4d7cdaae17304219179c2efafa5183a32fe5c4f7835a59e1212177af2fd6d88b87892d7d8a529e68e530dec3739a185dc56c28f0e8f77b016588c4bd4106f2c445f340c9ec4d8f618f7dc1741be067c5689c79bc0b2f8a73eec36ab8fe4d69a9f12c179cb6554f40ecfb06efe62a37e058d9d7c7a79bfc61c708f71c976443ecd9f89df77f02c64eafb76af0bde669a06436abc66fda04cc6da9a966fe16cf7da7ce6c267c8d74e3b1a9b09db419a0d52ca26b0f3b00e212d2312c0696e3e448d8644441f072ee8bed53861df320792680ff0087d460d524db562e363b8efd818b60f1274c9a7afc6b670eaa94980cc0e8b6ea5bd239e7808f1c214dec43bfb4131e9754ff83c99b954acbfb90c5b1cb7b1a297e28f1beeba26e28d3504dfc6b57e68885e5840c4f4746f280ce6b909fde4ab2ad11077ba15433ab6cfa42de6079de9210ee69097895e12ff41777dc94db76e41b59b88c1864f10be6d8aa80abad8018140a94e1c7a47e2a133f6caa56291caccd21237b122c30da82de2274aacb9ec476113daaae7bcd677a970c3c62b9b8c91cb21c9ab0108cfc4dee362f2346cafc43fcb67a0a37f2af638e1996f1e8d591aeb31c79482102dd0fa0ced6c3421e58bffa219566b735a4baf652df736a74183da30f9863876ff3da528971d67648c01a739bad1e026825b84906e2653a12ac77c24c1cd14369f426135c2029a5d9749be5f47c0d87fb953c2782a475dc406700c2b61fb7ab4faa8a3f23fc23aeb964677a1f7d55a2817b122cc9d120f5c3d751fe7fe950deb989fc13ee7d53d1052d322816b844685f122f7fa321bfb3499802aca9c564d41b39fbfff5ff777873ad1e40c509ac2016eaf08736334650d4bdb4a7c895033a16e1c44acced720fbde04982310639ea4e92bfe065ab16dd25edc4554137eeda0a37bc0a312558a8fc6f2f5c8e1a5b3d81bc75eade5a84b32f81a638fce94181bfd540a4a0b7411bd66fa199da76e10dac8abc04a3738e40ac28c4b6b0935e828b3800bd7d21f5f7ef46ee1cf05f12b6cf335c45fd4093df9d87852653a3717843f3693c240fff8ef9b26b76166ffec4b798490a9015d3ec7740465a0f42ebc2192fe2c5c74a4f428e7c9c86697aa05fb719345dd94a4784dcb9165b0f56f650d8242f89aafab44558a964671660da5aef2debc16172924bdd77cc06981478ac1859cdf0e2a24d5d3911906c7eec01d3a2f7cc6c0f05db395d7518073d7d65fa82269ca90fbed3b207beadc8ffb0547d3e321887fdeb208a9d70d1f6756c8fb7fb86fa05ac33a13c208960951922ded8cacf138b6ceed37245f960a7efddaa5d2111eeebae7de3a8704eac7ede1ca90077cc78d0b1c61bab486db05c7f728cb1a3983eb605c82ca1a36d5d4a90ecb7920862594699b3384b33fbdf22cb89ea98bad56b234cc403aa57dbd140cf0a35900d8a6d7948ef2ee5418db27a7499e37db44303ec8c5dba233c7faf1c73005a96a082c3ea5af948b3fed72386f2a460cbce949cbf9732282667ba64dba04728333f0ad5b6fa63e04b5373b8d8abb993841439aa51ef4ecff6893be4f1540985c3804afcb1d079651c03acd305da29de2b8ca4bd28cb422f11f28f43bbad5174ec7bb6dc7cb5b81afdcbbd5e9a84bda0aba6d06a92323eb34a9774c9ea3c8db37280fe7e2a1b8e9afb7844b30c07024ee283d2032df6cb3e4fe6acd1bc7dc69b75bc1015fc71ea0747a076a16101bfa4da39e647f98f753fd87d79a3fbdea6fe8739f460ca9cd2fda2a1b4596441140d42059c5e3b41e6202c8557009ead726912d8da3b728f4d98aa2cd32cb76cbacde8978f0f374fa66ecef2bbba1509d2c37df52274793a586439560be95b4ec9db4f8488165330fc97851fb2b7bbdea5e2765b2223d80783d675d52c115948fa10652869d91cd947ad9cdb27aea8c78f5ebb9e3ae6864f6fec3721412fa875d991017e3e6f56b7d440ccd1ef04daa1b896d947daea3d194591965396d0d257fe4bfb71c4b5c06a76d23fda40b603aba0557cac78062abdd6fc47647b5dcbb6fd8409ab67645dd40e28db17af59bf65694e7bd71a648461d006021cc1b76c4075cb9bd05b5f61bfa22767c6c967081f2554e046a120cedb0da59136a5d316bebd3401a91041d6d3503209e43746b9a3e5ea2dd2e64a15183f5921710659dc615545bead57d72e6336d393dd9a94c8ebfee9de0e1cb30cbb116e770f0464a2972e7eb2a8fda7dbf9acdb4bf7cb6890bf4dd2afaade4cb50e7d51fe4d95235cabb34eb63ebaff19f242834943945a4c525b4f9d5c2af0536cf3628525ca528ddf0cebf6f751313ecb64113c061f59f5b992d830242f88b9f06d8e9d5c7eb73f6f9991d393beaf70a53a83ec0c31074df4e6c99ab531ff2c2177902ab58c726b6e0ef1c69c43134760cb1b158e8bb0bcc15dc54d2a98999cdb78b33a4fa21121a46915888046e2c9d7dbc3632c4aba535b1522d7b6255d2a03cf1a216987c073e217dd997a989369a4e227d1e2b9f2462ad54cfbf31d591f9a798456f92a3097acd4d20199a4ad3e66a00faa9ce0539ba5547f107a164b6d4cc92088f64cb343651b43e67f0aa6bc0a0256d4a314cdf1906cec1857786cfe3a097ec9f3c7cd46929f1396aa9ac631a7925fe1e81179c43670d720e00ff4d5088ef3bf495c01f7b7c649e8bfa87c69749e6e2a51a99b478ff8ab7007ba5b3ca135df8699c448da129dc6140ce34bb672b05fe307bb1633c96e0dd84ae4dc1caf75b2e2b3180ced90eb79e76c1752b77ac5e3aef2f4bdef3c667df53dcb95ca1e18d6405f4b6310108c23cb2c6bfc0c17d9a9c62baa870e3fe7eff300201178fe00462244bbaadf3607e60380db37e3836035904116697af246e5a7253eaa194902f61246a76bfd41e7531bf4db33ca0a3cf8761780edb67e308c93ea5baeae71b368599f7563658c60381c3523999699e86095989b8ff6275dd20402e5b35b831efe8c4755709284c4ea09c6702b209864e63ade180ab0c7ecdb376c510c55bee53d33577a1ae32d0e6e8d714e9c6c40193ef22e771b521ffb0e246176cb09c30d28e98a439d935fd4d2f1be11548c9147425395e99bb7ee0ac880a43c9ae63ee086728e9db1d57e0324dec05f3be6ed1c4d192cf18e8a6704375ab9632ec065773a206b88eb2d1e36e428e4341ad6e300b1c49f636caafdd28b0546d6748b9211df0273ab27921f3a767efa4e0d38cffbbe2f53e77b7c888c5cd81369602d4d2b9ab3a92f8180306b7e339a18c73cec1a253c319c46e1b71ac8910a11df903cf6c4ff5b392a96b282f849819b47a919ebe9966bff73e202267cffed1deb3c9c9df6e88c4b292b8823e2efc111af20b6c25719a3db8e1451e710717d7412cbc08566d55923784fc0f4a70f903377d73b29033fc61a445fd57e2db1eb9be74334edf13dc5b9bdd9256e158ab3642b1ee3bf8b7e07747abd2114c4d8e45443e14480cd3b6998e6a4dec41290d854fe978bda36f2102506dfc7cb75950409bf174bc268e5e6f9192c9531ba98c0e66ebbb963736866e0f2ad703ee08ef9a5746e6dc76d17bcc192bf380e94a493319621e8adcbeefbe80bd0388f8dc84824471f040dbf72e1d1f068c77a695fc1dc47f18c70ee7e697f0106420e8ec0145b8670d4a3642d776ab7b90be5181a1189457afaf701afa3d4cb7e05219f9c031987f6d4c6cc49d8c5c0e6b67dc5b066f86a74d7e7bfc851af05f23d4fb477d6d4e6242bcf58f1d58a5f0f363e15833488fe98a4306efd560550eb4ee755c229f41d69b9947039ceb254fb64a24792c75be2a29c5058ce0cb89eb86647d657c146360943a29047e3fc3e84936d1d35bbc16885640d3510035eef02a36c582c86fb280292880518873d6bd44b8846bd14573c28ff95c7731bde4d3750dff967d336526773dc809657e589c023a96fb23bf79fd41539195abd3e21f538aba3da6b8b276dc69cb9938e57bfd19ee4ca34141a86fd509644c32dd237bdf26538f52ab287570e35ac0ae86e62dcbdc8a3f1c8ad4ac0a756226439b80ed2f3b5664e9093ba4514ad52da6f1f863b67f4a21ca1863662a266920dd7750dd863f22ede785775b89002daca5c8f00e31c0f9c4b2716481d5830269172e3596907aac87f4e81ef66cfd976e523905534c13fd9f006342d4ee39584f7c196f8c6db1d59693229e4010037302f978b2d8ab5079ea1ef4f431bb9e11ecacdf9f58ebfbc24c50f542911a9b950491bd4a6593aed15177981a94e0723a2dba5dd63fb87cf6d475bfa42d0df7d0fc5b3b1bf9621066c1f32a45061363c3175074fed91377556c63e104186f7609b5a06e4c3902a8404e295929d4d786c15a48be408d8745d9674873c5a7ad4756e41e64188016191e8da12aa4b30238b29e4d784e8950e808931f12424fe4dad6d8cb0bb55e8c0440e0148f04d50b06acd17af3a47854344b9a32a471853b0ded34b056bdc956318cc494f920330d23606698bde3caf30fafe14287fd4b2351bdaacf2cd76b30c90ef07a13dfd40e63f8ad23050b3819d08e6aeca55193b23b0e748b9b358aadb3ed7d1e2affac941f7be7fa35005ccfeb02929320d8db088245cf16f5bf811a90b37f8e30c7ee4c8ff0aa018a109e544f5c51e844152315da766f4c19033796bc9e21e981264f9f118947f1775acf373800c77a063be4473994d511c1e4919047f00e988c8912a8647fdde6c3a8a632026893d67839c20f25b7a3f64f52975be7c25a1877d77478b8b8134c57c6d2e043c6555bb6927142b972c3c70a3a7807f83c5f110fcba676248731ac51078580b1a2f605adb892e9b419f1d9509eb4fb9c5f3fd92e47dae7cb3596e6d04c182834e6ba1d37f139b4f89c95113a77c2c1f802b6ce1624cefca77918047582b694ee5d7da62d97efe090a09b6a3e6723cf09714adfa91bd7dce2f6ec35d53db92d7bc779d260372518d57c973cd07f890c6d9ca234116560c1773a7320aea412b04341c29b32254caed021817a12e9dc00740e76b13aad96dd43544ae873175c05982288134aa8460002515ba8779497ffad9699ad248c11556ec8228e2668845b3e64a2d202b6ca76fc89eaacaa6568ea1b9f0aa6b80ba0876ea58441feba47fc37a0d5a4a0c313bcd5a49de1317eb0498d3f17104c46e0674280be32f5c691b5fd9748e37fc0a00a12022559a411402365c37f74cdf90311cdd4d810929f2f8cfedd690ac1fc52e0d51303e43b070fcf8314537ec4a7d97aa19944a5496e919c72b9bbf0addea5855dc99893e0e6236960ee7cc9d496ddbbb040c9773739ae79d9a5cc95aa119a3210d5e18c7f3d9953050929fe2adedc7df10085c057f95473c6bde947df47a4d17c9a934b95aedf0bff303051428610cd57ed321dea4b0ce12684c0803ad5efacebb7a826a6be3732242d67839f86f99a17dcfe5befd3f903";
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig1), sig1));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig2), sig2));
        assertEquals(true, Arrays.areEqual(Hex.decode(expectedSig3), sig3));
    }

    public void testVerifySignatureSHA256()
    {
        XMSSParameters params = new XMSSParameters(4, new SHA256Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        byte[] msg1 = new byte[1024];

        for (int i = 0; i < 3; i++)
        {
            byte[] publicKey = xmss.exportPublicKey();
            xmss.sign(msg1);
            byte[] signature = xmss.sign(msg1);
            try
            {
                assertEquals(true, xmss.verifySignature(msg1, signature, publicKey));
            }
            catch (ParseException ex)
            {
                ex.printStackTrace();
                fail();
            }
            byte[] msg2 = new byte[1024];
            msg2[0] = 0x01;
            try
            {
                assertEquals(false, xmss.verifySignature(msg2, signature, publicKey));
            }
            catch (ParseException ex)
            {
                ex.printStackTrace();
                fail();
            }
        }
    }

    public void testVerifySignatureSHA512()
    {
        XMSSParameters params = new XMSSParameters(4, new SHA512Digest());
        XMSS xmss = new XMSS(params, new NullPRNG());
        xmss.generateKeys();
        byte[] msg1 = new byte[1024];

        for (int i = 0; i < 3; i++)
        {
            byte[] publicKey = xmss.exportPublicKey();
            xmss.sign(msg1);
            byte[] signature = xmss.sign(msg1);
            try
            {
                assertEquals(true, xmss.verifySignature(msg1, signature, publicKey));
            }
            catch (ParseException ex)
            {
                ex.printStackTrace();
                fail();
            }
            byte[] msg2 = new byte[1024];
            msg2[0] = 0x01;
            try
            {
                assertEquals(false, xmss.verifySignature(msg2, signature, publicKey));
            }
            catch (ParseException ex)
            {
                ex.printStackTrace();
                fail();
            }
        }
    }

    public void testImportStateSHA256()
        throws IOException, ClassNotFoundException
    {
        XMSSParameters params = new XMSSParameters(4, new SHA256Digest());
        XMSS xmss1 = new XMSS(params, new NullPRNG());
        xmss1.generateKeys();
        byte[] msg1 = new byte[1024];
        byte[] msg2 = new byte[2048];
        byte[] msg3 = new byte[3096];
        Arrays.fill(msg1, (byte)0xaa);
        Arrays.fill(msg2, (byte)0xbb);
        Arrays.fill(msg3, (byte)0xcc);
        byte[] signature1 = xmss1.sign(msg1);
        byte[] signature2 = xmss1.sign(msg2);
        byte[] exportedPrivateKey = xmss1.exportPrivateKey();
        byte[] exportedPublicKey = xmss1.exportPublicKey();
        byte[] signature3 = xmss1.sign(msg3);

        XMSS xmss2 = new XMSS(params, new NullPRNG());

            xmss2.importState(exportedPrivateKey, exportedPublicKey);

        byte[] signature4 = xmss2.sign(msg3);
        assertEquals(true, Arrays.areEqual(signature3, signature4));
        xmss2.generateKeys();
        try
        {
            assertEquals(true, xmss2.verifySignature(msg1, signature1, exportedPublicKey));
            assertEquals(true, xmss2.verifySignature(msg2, signature2, exportedPublicKey));
            assertEquals(true, xmss2.verifySignature(msg3, signature3, exportedPublicKey));
            assertEquals(false, xmss2.verifySignature(msg1, signature3, exportedPublicKey));
            assertEquals(false, xmss2.verifySignature(msg2, signature3, exportedPublicKey));
        }
        catch (ParseException ex)
        {
            ex.printStackTrace();
            fail();
        }
    }

    public void testImportKeysSHA512()
        throws IOException, ClassNotFoundException
    {
        XMSSParameters params = new XMSSParameters(4, new SHA512Digest());
        XMSS xmss1 = new XMSS(params, new NullPRNG());
        xmss1.generateKeys();
        byte[] msg1 = new byte[1024];
        byte[] msg2 = new byte[2048];
        byte[] msg3 = new byte[3096];
        Arrays.fill(msg1, (byte)0xaa);
        Arrays.fill(msg2, (byte)0xbb);
        Arrays.fill(msg3, (byte)0xcc);
        byte[] signature1 = xmss1.sign(msg1);
        byte[] signature2 = xmss1.sign(msg2);
        byte[] exportedPrivateKey = xmss1.exportPrivateKey();
        byte[] exportedPublicKey = xmss1.exportPublicKey();
        byte[] signature3 = xmss1.sign(msg3);

        XMSS xmss2 = new XMSS(params, new NullPRNG());

            xmss2.importState(exportedPrivateKey, exportedPublicKey);

        byte[] signature4 = xmss2.sign(msg3);
        assertEquals(true, Arrays.areEqual(signature3, signature4));
        xmss2.generateKeys();
        try
        {
            assertEquals(true, xmss2.verifySignature(msg1, signature1, exportedPublicKey));
            assertEquals(true, xmss2.verifySignature(msg2, signature2, exportedPublicKey));
            assertEquals(true, xmss2.verifySignature(msg3, signature3, exportedPublicKey));
            assertEquals(false, xmss2.verifySignature(msg1, signature3, exportedPublicKey));
            assertEquals(false, xmss2.verifySignature(msg2, signature3, exportedPublicKey));
        }
        catch (ParseException ex)
        {
            ex.printStackTrace();
            fail();
        }
    }

    public void testRandom()
        throws IOException, ClassNotFoundException
    {
        XMSSParameters params = new XMSSParameters(4, new SHA512Digest());
        XMSS xmss1 = new XMSS(params, new SecureRandom());
        xmss1.generateKeys();
        byte[] publicKey = xmss1.exportPublicKey();
        byte[] message = new byte[1024];

        for (int i = 0; i < 5; i++)
        {
            xmss1.sign(message);
        }
        byte[] signature = xmss1.sign(message);
        assertTrue(Arrays.areEqual(publicKey, xmss1.exportPublicKey()));
        try
        {
            xmss1.verifySignature(message, signature, publicKey);
        }
        catch (ParseException e)
        {
            e.printStackTrace();
            fail();
        }
        assertTrue(Arrays.areEqual(publicKey, xmss1.exportPublicKey()));
        xmss1.sign(message);
        byte[] privateKey7 = xmss1.exportPrivateKey();
        try
        {
            xmss1.verifySignature(message, signature, publicKey);
        }
        catch (ParseException e)
        {
            e.printStackTrace();
            fail();
        }
        assertTrue(Arrays.areEqual(privateKey7, xmss1.exportPrivateKey()));
        byte[] signature7 = xmss1.sign(message);

        xmss1.importState(privateKey7, publicKey);

        byte[] signature7AfterImport = xmss1.sign(message);
        assertTrue(Arrays.areEqual(signature7AfterImport, signature7));

        XMSSParameters params2 = new XMSSParameters(4, new SHA512Digest());
        XMSS xmss2 = new XMSS(params2, new SecureRandom());
        try
        {
            boolean valid = xmss2.verifySignature(message, signature7, publicKey);
            assertTrue(valid);
            valid = xmss2.verifySignature(message, signature, publicKey);
            assertTrue(valid);
        }
        catch (ParseException e)
        {
            e.printStackTrace();
            fail();
        }

        XMSS xmss3 = new XMSS(params, new NullPRNG());

            xmss3.importState(privateKey7, publicKey);

        byte[] signatureAgain = xmss3.sign(message);
        assertTrue(Arrays.areEqual(signatureAgain, signature7));
    }

    public void testBDSImport()
        throws IOException, ClassNotFoundException
    {
        XMSSParameters params = new XMSSParameters(4, new SHA256Digest());
        XMSS xmss = new XMSS(params, new SecureRandom());
        xmss.generateKeys();
        byte[] exportedPrivateKey = xmss.exportPrivateKey();
        byte[] exportedPublicKey = xmss.exportPublicKey();

        xmss.importState(exportedPrivateKey, exportedPublicKey);

        byte[] sig1 = xmss.sign(new byte[1024]);

        xmss.importState(exportedPrivateKey, exportedPublicKey);

        byte[] sig2 = xmss.sign(new byte[1024]);
        assertEquals(true, Arrays.areEqual(sig1, sig2));
        try
        {
            xmss.importState(exportedPrivateKey, exportedPublicKey);
        }
        catch (Exception ex)
        {
        }
        xmss.sign(new byte[1024]);
        byte[] sig3 = xmss.sign(new byte[1024]);
        assertEquals(false, Arrays.areEqual(sig1, sig3));
        try
        {
            xmss.importState(null, exportedPublicKey);
            fail();
        }
        catch (Exception ex)
        {
        }
        try
        {
            xmss.importState(exportedPrivateKey, null);
            fail();
        }
        catch (Exception ex)
        {
        }
    }
}
