#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

from fenrirscreenreader.core.i18n import _
from fenrirscreenreader.utils import word_utils
from fenrirscreenreader.core import debug


class command:
    def __init__(self):
        pass

    def initialize(self, environment):
        self.env = environment

    def shutdown(self):
        pass

    def get_description(self):
        return _("current word.")

    def run(self):
        self.env["runtime"][
            "CursorManager"
        ].enter_review_mode_curr_text_cursor()

        (
            self.env["screen"]["newCursorReview"]["x"],
            self.env["screen"]["newCursorReview"]["y"],
            curr_word,
            end_of_screen,
            line_break,
        ) = word_utils.get_current_word(
            self.env["screen"]["newCursorReview"]["x"],
            self.env["screen"]["newCursorReview"]["y"],
            self.env["screen"]["new_content_text"],
        )

        # Check if we're in table mode and provide table-aware output
        is_table_mode = self.env["runtime"]["TableManager"].is_table_mode()
        self.env["runtime"]["DebugManager"].write_debug_out(
            f"review_curr_word: is_table_mode={is_table_mode}",
            debug.DebugLevel.INFO
        )
        if is_table_mode:
            # Get current cell info using internal column tracking
            table_info = (
                self.env["runtime"]["TableManager"].get_current_table_cell_info()
            )
            if table_info:
                # Announce with table context - cell content first, then header
                output_text = (
                    f"{table_info['cell_content']} {table_info['column_header']}"
                )
                self.env["runtime"]["OutputManager"].present_text(
                    output_text, interrupt=True, flush=False
                )
                return  # Exit early for table mode
            else:
                # Fallback to regular word announcement
                if curr_word.isspace():
                    self.env["runtime"]["OutputManager"].present_text(
                        _("blank"), interrupt=True, flush=False
                    )
                else:
                    self.env["runtime"]["OutputManager"].present_text(
                        curr_word, interrupt=True, flush=False
                    )
        else:
            # Regular word announcement
            if curr_word.isspace():
                self.env["runtime"]["OutputManager"].present_text(
                    _("blank"), interrupt=True, flush=False
                )
            else:
                self.env["runtime"]["OutputManager"].present_text(
                    curr_word, interrupt=True, flush=False
                )
        if end_of_screen:
            if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "review", "end_of_screen"
            ):
                self.env["runtime"]["OutputManager"].present_text(
                    _("end of screen"),
                    interrupt=True,
                    sound_icon="EndOfScreen",
                )
        if line_break:
            if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "review", "line_break"
            ):
                self.env["runtime"]["OutputManager"].present_text(
                    _("line break"), interrupt=False, sound_icon="EndOfLine"
                )

    def set_callback(self, callback):
        pass
