/* SPDX-License-Identifier: GPL-2.0-only */
/* Copyright (C) 2023--2024 Intel Corporation */

#ifndef IPU6_PLATFORM_ISYS_CSI2_REG_H
#define IPU6_PLATFORM_ISYS_CSI2_REG_H

#include <linux/bits.h>

#define CSI_REG_BASE			0x220000
#define CSI_REG_PORT_BASE(id)		(CSI_REG_BASE + (id) * 0x1000)

/* CSI Port Genral Purpose Registers */
#define CSI_REG_PORT_GPREG_SRST                 0x0
#define CSI_REG_PORT_GPREG_CSI2_SLV_REG_SRST    0x4
#define CSI_REG_PORT_GPREG_CSI2_PORT_CONTROL    0x8

/*
 * Port IRQs mapping events:
 * IRQ0 - CSI_FE event
 * IRQ1 - CSI_SYNC
 * IRQ2 - S2M_SIDS0TO7
 * IRQ3 - S2M_SIDS8TO15
 */
#define CSI_PORT_REG_BASE_IRQ_CSI               0x80
#define CSI_PORT_REG_BASE_IRQ_CSI_SYNC          0xA0
#define CSI_PORT_REG_BASE_IRQ_S2M_SIDS0TOS7     0xC0
#define CSI_PORT_REG_BASE_IRQ_S2M_SIDS8TOS15    0xE0

#define CSI_PORT_REG_BASE_IRQ_EDGE_OFFSET	0x0
#define CSI_PORT_REG_BASE_IRQ_MASK_OFFSET	0x4
#define CSI_PORT_REG_BASE_IRQ_STATUS_OFFSET	0x8
#define CSI_PORT_REG_BASE_IRQ_CLEAR_OFFSET	0xc
#define CSI_PORT_REG_BASE_IRQ_ENABLE_OFFSET	0x10
#define CSI_PORT_REG_BASE_IRQ_LEVEL_NOT_PULSE_OFFSET	0x14

#define IPU6SE_CSI_RX_ERROR_IRQ_MASK		GENMASK(18, 0)
#define IPU6_CSI_RX_ERROR_IRQ_MASK		GENMASK(19, 0)

#define CSI_RX_NUM_ERRORS_IN_IRQ		20
#define CSI_RX_NUM_IRQ				32

#define IPU_CSI_RX_IRQ_FS_VC(chn)	(1 << ((chn) * 2))
#define IPU_CSI_RX_IRQ_FE_VC(chn)	(2 << ((chn) * 2))

/* PPI2CSI */
#define CSI_REG_PPI2CSI_ENABLE				0x200
#define CSI_REG_PPI2CSI_CONFIG_PPI_INTF			0x204
#define PPI_INTF_CONFIG_NOF_ENABLED_DLANES_MASK		GENMASK(4, 3)
#define CSI_REG_PPI2CSI_CONFIG_CSI_FEATURE		0x208

enum CSI_PPI2CSI_CTRL {
	CSI_PPI2CSI_DISABLE = 0,
	CSI_PPI2CSI_ENABLE = 1,
};

/* CSI_FE */
#define CSI_REG_CSI_FE_ENABLE                   0x280
#define CSI_REG_CSI_FE_MODE                     0x284
#define CSI_REG_CSI_FE_MUX_CTRL                 0x288
#define CSI_REG_CSI_FE_SYNC_CNTR_SEL            0x290

enum CSI_FE_ENABLE_TYPE {
	CSI_FE_DISABLE = 0,
	CSI_FE_ENABLE = 1,
};

enum CSI_FE_MODE_TYPE {
	CSI_FE_DPHY_MODE = 0,
	CSI_FE_CPHY_MODE = 1,
};

enum CSI_FE_INPUT_SELECTOR {
	CSI_SENSOR_INPUT = 0,
	CSI_MIPIGEN_INPUT = 1,
};

enum CSI_FE_SYNC_CNTR_SEL_TYPE {
	CSI_CNTR_SENSOR_LINE_ID = BIT(0),
	CSI_CNTR_INT_LINE_PKT_ID = ~CSI_CNTR_SENSOR_LINE_ID,
	CSI_CNTR_SENSOR_FRAME_ID = BIT(1),
	CSI_CNTR_INT_FRAME_PKT_ID = ~CSI_CNTR_SENSOR_FRAME_ID,
};

/* CSI HUB General Purpose Registers */
#define CSI_REG_HUB_GPREG_SRST			(CSI_REG_BASE + 0x18000)
#define CSI_REG_HUB_GPREG_SLV_REG_SRST		(CSI_REG_BASE + 0x18004)

#define CSI_REG_HUB_DRV_ACCESS_PORT(id)	(CSI_REG_BASE + 0x18018 + (id) * 4)
#define CSI_REG_HUB_FW_ACCESS_PORT_OFS		0x17000
#define CSI_REG_HUB_FW_ACCESS_PORT_V6OFS	0x16000
#define CSI_REG_HUB_FW_ACCESS_PORT(ofs, id)	\
					(CSI_REG_BASE + (ofs) + (id) * 4)

enum CSI_PORT_CLK_GATING_SWITCH {
	CSI_PORT_CLK_GATING_OFF = 0,
	CSI_PORT_CLK_GATING_ON = 1,
};

#define CSI_REG_BASE_HUB_IRQ                        0x18200

#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_EDGE			0x238200
#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_MASK			0x238204
#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_STATUS			0x238208
#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_CLEAR			0x23820c
#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_ENABLE			0x238210
#define IPU6_REG_ISYS_CSI_TOP_CTRL0_IRQ_LEVEL_NOT_PULSE		0x238214

#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_EDGE			0x238220
#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_MASK			0x238224
#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_STATUS			0x238228
#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_CLEAR			0x23822c
#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_ENABLE			0x238230
#define IPU6_REG_ISYS_CSI_TOP_CTRL1_IRQ_LEVEL_NOT_PULSE		0x238234

/* MTL IPU6V6 irq ctrl0 & ctrl1 */
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_EDGE			0x238700
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_MASK			0x238704
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_STATUS		0x238708
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_CLEAR			0x23870c
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_ENABLE		0x238710
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL0_IRQ_LEVEL_NOT_PULSE	0x238714

#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_EDGE			0x238720
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_MASK			0x238724
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_STATUS		0x238728
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_CLEAR			0x23872c
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_ENABLE		0x238730
#define IPU6V6_REG_ISYS_CSI_TOP_CTRL1_IRQ_LEVEL_NOT_PULSE	0x238734

/*
 * 3:0 CSI_PORT.irq_out[3:0] CSI_PORT_CTRL0 IRQ outputs (4bits)
 * [0] CSI_PORT.IRQ_CTRL0_csi
 * [1] CSI_PORT.IRQ_CTRL1_csi_sync
 * [2] CSI_PORT.IRQ_CTRL2_s2m_sids0to7
 * [3] CSI_PORT.IRQ_CTRL3_s2m_sids8to15
 */
#define IPU6_ISYS_UNISPART_IRQ_CSI2(port)		\
				   (0x3 << ((port) * IPU6_CSI_IRQ_NUM_PER_PIPE))

/*
 * ipu6se support 2 front ends, 2 port per front end, 4 ports 0..3
 * sip0 - 0, 1
 * sip1 - 2, 3
 * 0 and 2 support 4 data lanes, 1 and 3 support 2 data lanes
 * all offset are base from isys base address
 */

#define CSI2_HUB_GPREG_SIP_SRST(sip)			(0x238038 + (sip) * 4)
#define CSI2_HUB_GPREG_SIP_FB_PORT_CFG(sip)		(0x238050 + (sip) * 4)

#define CSI2_HUB_GPREG_DPHY_TIMER_INCR			0x238040
#define CSI2_HUB_GPREG_HPLL_FREQ			0x238044
#define CSI2_HUB_GPREG_IS_CLK_RATIO			0x238048
#define CSI2_HUB_GPREG_HPLL_FREQ_ISCLK_RATE_OVERRIDE	0x23804c
#define CSI2_HUB_GPREG_PORT_CLKGATING_DISABLE		0x238058
#define CSI2_HUB_GPREG_SIP0_CSI_RX_A_CONTROL		0x23805c
#define CSI2_HUB_GPREG_SIP0_CSI_RX_B_CONTROL		0x238088
#define CSI2_HUB_GPREG_SIP1_CSI_RX_A_CONTROL		0x2380a4
#define CSI2_HUB_GPREG_SIP1_CSI_RX_B_CONTROL		0x2380d0

#define CSI2_SIP_TOP_CSI_RX_BASE(sip)		(0x23805c + (sip) * 0x48)
#define CSI2_SIP_TOP_CSI_RX_PORT_BASE_0(port)	(0x23805c + ((port) / 2) * 0x48)
#define CSI2_SIP_TOP_CSI_RX_PORT_BASE_1(port)	(0x238088 + ((port) / 2) * 0x48)

/* offset from port base */
#define CSI2_SIP_TOP_CSI_RX_PORT_CONTROL		0x0
#define CSI2_SIP_TOP_CSI_RX_DLY_CNT_TERMEN_CLANE	0x4
#define CSI2_SIP_TOP_CSI_RX_DLY_CNT_SETTLE_CLANE	0x8
#define CSI2_SIP_TOP_CSI_RX_DLY_CNT_TERMEN_DLANE(lane)	(0xc + (lane) * 8)
#define CSI2_SIP_TOP_CSI_RX_DLY_CNT_SETTLE_DLANE(lane)	(0x10 + (lane) * 8)

#endif /* IPU6_ISYS_CSI2_REG_H */
