%feature("docstring") OT::UserDefinedSpectralModel
R"RAW(Spectral model defined by the User.

Parameters
----------
frequency : :class:`~openturns.RegularGrid`
    The frequency grid :math:`(-f_c, \dots, f_c)` with step :math:`\delta f` on which the model is built.
densityCollectionFunction : :class:`~openturns.HermitianMatrixCollection`
    Collection of hermitian matrices :math:`\in \mathbb{M}_d(\mathbb{C})` which are the images of each point of the frequency grid through the density spectral function.

Notes
-----
We consider a multivariate process :math:`X: \Omega \times \cD \rightarrow \Rset^d` of dimension :math:`d` where :math:`\cD \in \Rset^n`. We only treat here the case where the domain is of dimension 1: :math:`\cD \in \Rset` (:math:`n=1`).

We denote :math:`(\vect{t}_0, \dots, \vect{t}_{N-1})` the time stamps of the time grid.

:math:`X` is supposed to be a stationary second order process and we suppose that its spectral density function :math:`S : \Rset \rightarrow \mathcal{H}^+(d)` defined by:

.. math::
    :label: fourier_transform

    \forall \vect{f} \in \Rset^n, \,S(\vect{f}) = \int_{\Rset^n}\exp\left\{  -2i\pi <\vect{f},\vect{\tau}> \right\} C^{stat}(\vect{\tau})\di{\vect{\tau}}

exists, where :math:`\mathcal{H}^+(d) \in \mathcal{M}_d(\Cset)` is the set of *d*-dimensional positive definite hermitian matrices.

We get a piecewise constant function on :math:`[-f_c,f_c]`, where the intervals on which the density spectral function is constant are centered on the points of the frequency grid, of length :math:`\delta f`. 

Then, it is  possible to evaluate the spectral density function for a given frequency thanks to the method *computeSpectralDensity*: if the frequency is not inside the interval :math:`[-f_c,f_c]`, OpenTURNS returns an exception. Otherwise, it returns the hermitian matrix of the subinterval of :math:`[-f_c,f_c]` that contains the given frequency.

Examples
--------
Create the frequency grid:

>>> import openturns as ot
>>> from math import exp
>>> fmin = 0.1
>>> df = 0.5
>>> N = int((10.0 - fmin)/ df)
>>> myFrequencyGrid =  ot.RegularGrid(fmin, df, N)

Define the spectral function:

>>> def s(f):
...     if(f <= 5.0):
...         return 1.0
...     else:
...         x = f - 5.0
...         return exp(-2.0 * x * x)

Create the collection of HermitianMatrix:

>>> myCollection = ot.HermitianMatrixCollection()
>>> for k in range(N):
...     frequency = myFrequencyGrid.getValue(k)
...     matrix = ot.HermitianMatrix(1)
...     matrix[0, 0] = s(frequency)
...     myCollection.add(matrix)

Create the spectral model:

>>> mySpectralModel = ot.UserDefinedSpectralModel(myFrequencyGrid, myCollection)
)RAW"

// ---------------------------------------------------------------------

%define OT_UserDefinedSpectralModel_get_frequency_grid_doc
"Return the frequency grid associated to the model.

Returns
-------
grid : :class:`~openturns.RegularGrid`
    The frequency grid of the model."

%enddef
%feature("docstring") OT::UserDefinedSpectralModel::getFrequencyGrid
OT_UserDefinedSpectralModel_get_frequency_grid_doc

// ---------------------------------------------------------------------

%define OT_UserDefinedSpectralModel_set_frequency_grid_doc
"Set the frequency grid associated to the model.

Parameters
----------
grid : :class:`~openturns.RegularGrid`
    The frequency grid of the model."

%enddef
%feature("docstring") OT::UserDefinedSpectralModel::setFrequencyGrid
OT_UserDefinedSpectralModel_set_frequency_grid_doc

