; Tests of the SHA-224 formalization
;
; Copyright (C) 2018-2019 Kestrel Institute
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SHA2")

(include-book "sha-224")
(include-book "kestrel/bv-lists/string-to-bits" :dir :system)
(include-book "kestrel/utilities/hex-string-to-bytes" :dir :system)
(include-book "std/testing/assert-equal" :dir :system)

;;
;; Examples (see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA224.pdf)
;;

;; Example 1, bit version
(acl2::assert-equal (sha-224 (acl2::string-to-bits "abc"))
                    (acl2::hex-string-to-bytes! "23097D223405D8228642A477BDA255B32AADBCE4BDA0B3F7E36C9DA7"))

;; Example 2, bit version
(acl2::assert-equal (sha-224 (acl2::string-to-bits "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"))
                    (acl2::hex-string-to-bytes! "75388B16512776CC5DBA5DA1FD890150B0C6455CB4F58B1952522525"))

;; Example 1, byte version
(acl2::assert-equal (sha-224-bytes (acl2::string-to-bytes "abc"))
                    (acl2::hex-string-to-bytes! "23097D223405D8228642A477BDA255B32AADBCE4BDA0B3F7E36C9DA7"))

;; Example 2, byte version
(acl2::assert-equal (sha-224-bytes (acl2::string-to-bytes "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"))
                    (acl2::hex-string-to-bytes! "75388B16512776CC5DBA5DA1FD890150B0C6455CB4F58B1952522525"))

;;
;; Tests from the .zip file https://github.com/coruus/nist-testvectors/raw/master/csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip:
;;

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! ""))
                    (acl2::hex-string-to-bytes! "d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "84"))
                    (acl2::hex-string-to-bytes! "3cd36921df5d6963e73739cf4d20211e2d8877c19cff087ade9d0e3a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "5c7b"))
                    (acl2::hex-string-to-bytes! "daff9bce685eb831f97fc1225b03c275a6c112e2d6e76f5faf7a36e6"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "51ca3d"))
                    (acl2::hex-string-to-bytes! "2c8959023515476e38388abb43599a29876b4b33d56adc06032de3a2"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "6084347e"))
                    (acl2::hex-string-to-bytes! "ae57c0a6d49739ba338adfa53bdae063e5c09122b77604780a8eeaa3"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "493e14623c"))
                    (acl2::hex-string-to-bytes! "7f631f295e024e74552083245ca8f988a3fb65680ae97c3040d2e65c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "d729d8cd1631"))
                    (acl2::hex-string-to-bytes! "342e8e6b23c1c6a54910631f098e08e836259c57e49c1b1d023d166d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "cbf2061e10faa5"))
                    (acl2::hex-string-to-bytes! "3aa702b1b66dc57d7aec3ccdbdfbd88592d7520f843ba5d0fa481168"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "5f77b3664823c33e"))
                    (acl2::hex-string-to-bytes! "bdf21ff325f754157ccf417f4855360a72e8fd117d28c8fe7da3ea38"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "10713b894de4a734c0"))
                    (acl2::hex-string-to-bytes! "03842600c86f5cd60c3a2147a067cb962a05303c3488b05cb45327bd"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "006470d57dad9893dc03"))
                    (acl2::hex-string-to-bytes! "c90026cda5ad24115059c62ae9add57793ade445d4742273288bbce7"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "6f29ca274190400720bba2"))
                    (acl2::hex-string-to-bytes! "ac53157947aa4b2a19089182382a4363d182dd8e4ca79cd8571390be"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "17e8556176fcca2addbdde29"))
                    (acl2::hex-string-to-bytes! "cc6ad0488db0222066f740557b5758a19b30372b302332295d8c3aff"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "dbf163601db9a122a4026824de"))
                    (acl2::hex-string-to-bytes! "9849845f4e47e1ece9a1c1e01a0d896ffea61c6c8894a75a11ce5f49"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "5e1ef2ad86ceaf5439fe87d2ec9b"))
                    (acl2::hex-string-to-bytes! "223c5d5d4a0116b32cea044f9af0fe44babea1c5ab201502591bcd5f"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "65f3b9866fb8002b53cfaf806f702f"))
                    (acl2::hex-string-to-bytes! "b1e0806a218d593821fde8e9eacc44ab5287c32209a94f011ab66b75"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "b776708ffb91b3515ac46598ab9fa796"))
                    (acl2::hex-string-to-bytes! "427311b1d7ab2488791c4deeb4251d783fe5f9806bfdfb5188c5443d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "a4bc10b1a62c96d459fbaf3a5aa3face73"))
                    (acl2::hex-string-to-bytes! "d7e6634723ac25cb1879bdb1508da05313530419013fe255967a39e1"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "9e8f3c6645c1749b55c50d2018ce40dc2427"))
                    (acl2::hex-string-to-bytes! "2f5a583bf588c8988a572d128a95bea5ef1b66780a7d4be9c29efc31"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "2db6d207c0b7d9117f24d78ee59abf2f316978"))
                    (acl2::hex-string-to-bytes! "35681fce28307cae19522c23cbd4a77969347f7d8ee4a3088ba90ada"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "3df5e7f399f6dd61a12a9d4e9464fc4997c1f37b"))
                    (acl2::hex-string-to-bytes! "a3e68076e30751085a843a6cbfbf0f3dee63d9c4219c914372e50b28"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "65781d018f27ca0c72a9fa9ab4648ed369646dd3ce"))
                    (acl2::hex-string-to-bytes! "d15ef0d872d02da6427b8d0349dea2f204e67133b7365b4b150efc3c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "af48eeddd93fee69d1bd7de428a63986011d10945eaf"))
                    (acl2::hex-string-to-bytes! "b89d428ee42e397cf11029ecbb27baddd036c8938f51c8ab56b875ac"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "df2bf0d5f9c994ac69d78baa0d512eceb74d8a047531c1"))
                    (acl2::hex-string-to-bytes! "db8e1ce68c8c6b84d6db755c2b8bf54f3c4b081a881efcddaf303294"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "48d2f20955ea2d13433c20bc0404eb2e6ad79ed28f7cb4c0"))
                    (acl2::hex-string-to-bytes! "3617cc3179f8b59adce181eebeed5e2763f62650949224a67e53694b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "218f74a42d3a47ef3b806601fba024b078cbff4e4b85772e0e"))
                    (acl2::hex-string-to-bytes! "b5f40b95dcc363b97e9d00b67c5d7c37f17ab563297d2d67a4df20c9"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "ef55b1e797000b04fcdb9b3021b09327e3b4e269d20cabdf418f"))
                    (acl2::hex-string-to-bytes! "827b223d51240c2e3271c534c19c5637b6fe10083e85bcf06761ef21"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "96df4387dc2c40297043bea36483f65e4eb1e07e93359cb7e68610"))
                    (acl2::hex-string-to-bytes! "98e430a63fcdedafc9419010f7f59a4d816a45b4f973beb62530ff8c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "3ec0aa8d30d5ed825b77dc7095f421b1e608158797a377ff8bed641b"))
                    (acl2::hex-string-to-bytes! "3108321eb7ff857f6aae69101b937f32a51ea279a6c14ba5232ac8c1"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "8b0239712039f077ce323b35f4e306787b9b35270096e57735cff45d84"))
                    (acl2::hex-string-to-bytes! "a5c740d3ce46bb2e0a048488f2b0605c6d0ca0ea2f382d043d13db97"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "044be30167a9758c46c727921dc4eb4e0dcb965623423e6fdd44e7a4ea52"))
                    (acl2::hex-string-to-bytes! "6eb78313c743ea8769d8340f284dda6ded64a1db64392f21abb82c5c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "57f6118bacce47ecc31ce8b0c083d3c9219e0dbe9e4fbea154537c41231acc"))
                    (acl2::hex-string-to-bytes! "0dbb53c866d63af44c222c76c825df0e379dcedfb958db03b6fd29a5"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "fe1f0fb02c9011f4c8c5905934ed15136771737ce31c5859e67f235fe594f5f6"))
                    (acl2::hex-string-to-bytes! "bbeaacc632c2a3db2a9b47f157ab54aa27776c6e74cf0bcaa91b06d5"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "14fb01ae9d6015ecb3e56d6ecdfa4bc0533186adf8457f5e4a5c57c687895f3db3"))
                    (acl2::hex-string-to-bytes! "178272c7d7cc71b15074c27e3b7997d4a3ba99626986a1a16cf30030"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "ff6c49712f044f4063c14125c0cdfba18ed8b7138453768a45dfa2d82a05f1e84227"))
                    (acl2::hex-string-to-bytes! "403284c888a7280bc8bfc25f0c34182cd378306a21a1404d4e1c40cf"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "f900bd7e0117247f97c8fc7a665c76a35f571c3366571d6c4a3ee5d7fb93f1d1f726e2"))
                    (acl2::hex-string-to-bytes! "48235b9820d66d8885faabf6a9ede63ba2a21b6177e987a33242373e"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "42d38188ac49440cfefb77db975e083e6b22348c4c67f0f8692e88ad140d861dc828d595"))
                    (acl2::hex-string-to-bytes! "615344f890e5bcf71b5efe39de1fc942ba1fe30dd9e9146adb6a41bf"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "74fdd7d958b8ae7c2c3c5cff4266dfb2b3b842c9f59ecbbcaff575edcbcda08ccd6e08b764"))
                    (acl2::hex-string-to-bytes! "66d7d6c54fc7775a0ba845ba3e11719fa535b9289f20b098c5f7a342"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "934416dd0581e22f2bfbece7bb64afe820451fa21342df7e6f9fb37c4103381a1f7cd379bcc4"))
                    (acl2::hex-string-to-bytes! "fae8f1aa22def4dbaa814c5b0babdec43394951792c937050d2963a6"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "102401c84a716ae72579c6ae79c359ea309ffd95abffae4c61884c03c9e99df77b6c92e492cacb"))
                    (acl2::hex-string-to-bytes! "8f34812d57a16ef8a51ad987660c5f8623e0fa9d89846e28d46d14d9"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "79bc8fb60f85d15a2386566e3e7314df284533085add1c7bb6ead3ff760c86d5633a66404761b544"))
                    (acl2::hex-string-to-bytes! "65c54014cfa30f0bc27d1c6efa96ae8481f4c2505bff272956eab0df"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "db3121ea71294983b185207a9d8de3e484a66c0431bf07c962eb82977c4f834b7c3f1e7931a4a7f7a9"))
                    (acl2::hex-string-to-bytes! "9316d2f021c2913d63a7e66924c87c161c3cfde0ea7ba07f54772862"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "0dd51aa660c5cb4b7f78c46852c1db8707ab451c1367b6187388c8bb3873a1aa4210d0414cc6792a29a7"))
                    (acl2::hex-string-to-bytes! "31989e7a62a5132a5070d77250d8904bb82d457dc63469d06b50185e"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "487fd2e5b694b7071d3789a258a51e8604dc0d3e8f5d62f39131968e602abe1ddf6b0278962a512408b553"))
                    (acl2::hex-string-to-bytes! "e798683438284626d710877d9eea3a0e02f349fc43acb7f9f8f9e81c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "11183bdebfef58e4da5b1cb73be0d30b20da304d8659d921da2e270fd14626799537e4d12119e809ee97004a"))
                    (acl2::hex-string-to-bytes! "96870657d6cb668be3995aa8bd31df77840d1d1915d72482e83b6b2c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "a239de5c8e2644e8f030d94d98f1a30664e6fd961dc2977a9c08be5c31d8de89450945a53d79299ea2a1edde7f"))
                    (acl2::hex-string-to-bytes! "e99743d4fd26c8800c36a67b6762247c29da6b62794123c59de06dc0"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "917c4577aa6b0f9df49999fc1c958cb09b7fd5fc80be949670f03545eb27dcaed052076b24f96f5e0f2e2f4527c0"))
                    (acl2::hex-string-to-bytes! "7ecd693d4d9cf43929464698efa0bac33c2e1424f816edc769260978"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "c3f1e735a6741aa481ad577a98dbac1f03cc80ea0dae1b94db2369ed4e93facd29c64e4e77b25038279120bdfa3715"))
                    (acl2::hex-string-to-bytes! "86f0d89d8e14fd8b6606412d71a7a54a347b304ea5d49c208f2266ab"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "de4fbfd553cdf37019f25afa82dc6b9970f4bb1ebbc37f80d3084c88a70722cdc523a9e3c2afbad0dc0221bfdec9a2f9"))
                    (acl2::hex-string-to-bytes! "4c5262acb4a2a44eaa9bc6757024fb202ef4d5a7a16fa37252a422b5"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "db2e2eb636610cf42e9b33433acce1b3b925949f297dd83199f45d2861d64cd910c2db74a60b2089045e22cba0a536137d"))
                    (acl2::hex-string-to-bytes! "16bf4e45bcdc60447c68dcb30e6b08f55ce9f4124a29cf1f9a9d065d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "a8e729d336d5d6ac50e1e22f0b193b66e26042fc6459214129875e740ab2b142918c138aaf941863ad3b7e6065450613b273"))
                    (acl2::hex-string-to-bytes! "452bf2e5ebfc4e451cc434bc09e2a10032eed0b7627cf55e7e5ed0e2"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "d05317d4b535f9d10f739d0c2dedf3ffb090c1ad9d205089b1346693f58273c4925c0face57ba45ad6fc687c66a88fc78878be"))
                    (acl2::hex-string-to-bytes! "4f03c439e097b51b00e314f675937c4d911505859fb7ab16adc65e44"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "26bb4ed4f0424c60fe4212ff8c955e89e2f553a7d7701be59416d2089af59fa1074724e214e919b1e30f33fb78374b4b055bbc9b"))
                    (acl2::hex-string-to-bytes! "e7c899e27009d4dc77c2d300f191b757e52c9e7eac4b023bfab2b52a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "f015ec83944f03292463c4345fdb1c26d1ea07645facbc9520ae244b6eb191e53dabadb4ac0fb15cda4ed77dfb9e1193abfafb1b81"))
                    (acl2::hex-string-to-bytes! "459e40b3fbd612912f0217c60099379ce077cd02505871b0c9c14e7a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "0786706f680c27b792d054faa63f499a8e6b5ddb90502946235bf74c022d772c809cb4171bfa4791539aca1abd91900e53ba93ca0efd"))
                    (acl2::hex-string-to-bytes! "fadebab7c3d0fb8e97e429b79083087735e4ab385a789521260ef3ad"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "445e8698eeb8accbaac4ffa7d934fffd16014a430ef70f3a9174c6cfe96d1e3f6ab1377f4a7212dbb30146dd17d9f470c4dffc45b8e871"))
                    (acl2::hex-string-to-bytes! "4c7ae028c0fe61f2a9cada61fae30685b77f04c6442576e912af9fa6"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "52839f2f0853a30df14ec897a1914c685c1ac21470d00654c8c37663bfb65fa732dbb694d9dd09ced723b48d8f545846ba168988b61cc724"))
                    (acl2::hex-string-to-bytes! "2f755a57674b49d5c25cb37348f35b6fd2de2552c749f2645ba63d20"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "5fe8c2072d8900287ccaf07f3f66b0c22acd3e0bb91d9573754e19e373ac35271d8b43443436ac0c162850ef3d7f281409ad29a9bf716c77d1"))
                    (acl2::hex-string-to-bytes! "42909757f6e229f69f04cc7a863c4e70e48c7c3575057b455c959775"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "e8064d83f3d643af8718c87e3ccd6a9733685eac61d572a22ab943f232fcb04f70858e8984449db14a76bb7eaf2458efc3ed2a32100622c52b7f"))
                    (acl2::hex-string-to-bytes! "1a1d8ed54cb45c97bc970754b43eb93d9eabde4c7b07f76ad82d8ede"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "87c9a517e28d1bb54ad20fca76460efd894d7786e68ee8d746b2f68208682157c8ad06cc324ad7a3189e09c6c39d4c768719c0a49a41669f2767d5"))
                    (acl2::hex-string-to-bytes! "605977cf87b9b309bbddaaa64e528ace66b04df9f72c0e7ec88be1da"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "59fdac3b6b32039291801c7d6f46ede8d26dc5b7a192e007116739b617569f2523680b3c0b6631af453e55805aa760c6970833ac06963bbc9dbd455e"))
                    (acl2::hex-string-to-bytes! "e9f0cb1dc8337e906385892f2348a8ba4412318ecad9b96e3711531f"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "30350a4df0b58ff49c0fa09e426fcd7007b290c760c825c1855d9b0023b82caa51e3cab4c60cfa61492be50568e5ac0f6db0fd468e39e4536403e3809f"))
                    (acl2::hex-string-to-bytes! "776cc6636c02408fbf65ace73ae80017108b917c16c5a912fd860241"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "ef797a0d43c30b4fe1014bdb9420879c2ff845d27e73d55a7df22930c8ece73253d8bb265b4ef2ff9c69455cc56ff25229b4126bb7bb26ee2c9ff36187b1"))
                    (acl2::hex-string-to-bytes! "f5b9ffb102affac352a4a535a00f89b06c268cf4881d712668906025"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "716944de41710c29b659be10480bb25a351a39e577ee30e8f422d57cf62ad95bda39b6e70c61426e33fd84aca84cc7912d5eee45dc34076a5d2323a15c7964"))
                    (acl2::hex-string-to-bytes! "61645ac748db567ac862796b8d06a47afebfa2e1783d5c5f3bcd81e2"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "a3310ba064be2e14ad32276e18cd0310c933a6e650c3c754d0243c6c61207865b4b65248f66a08edf6e0832689a9dc3a2e5d2095eeea50bd862bac88c8bd318d"))
                    (acl2::hex-string-to-bytes! "b2a5586d9cbf0baa999157b4af06d88ae08d7c9faab4bc1a96829d65"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "f149e41d848f59276cfddd743bafa9a90e1ee4a263a118142b33e3702176ef0a59f8237a1cb51b42f3ded6b202d9af0997898fdd03cf60bda951c514547a0850cec25444ae2f24cb711bfbafcc3956c941d3de69f155e3f8b10f06db5f37359b772ddd43e1035a0a0d3db33242d5843033833b0dd43b870c6bf60e8deab55f317cc3273f5e3ba747f0cb65050cb7228796210d9254873643008d45f29cfd6c5b060c9a"))
                    (acl2::hex-string-to-bytes! "9db6dc3a23abd7b6c3d72c38f4843c7de48a71d0ba91a86b18393e5f"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "c39147fad02c6ed5876eb3257837c3f456008694fac94932aa521bae9c8e77abb65524e298595abbdc5b261e9c9f0f86359dfa584bf94b740eb54d09bba6d4ad652421adf50605a170ce4a4478204e831679f1d4b61db1c9735114e595cb47ae58670403f01bb8b0c92de64013a3c6137efc33b0421dc45b638e38eb33e617d61884968e8f80bb1071e1c3b97611c15cf78e8793f8e1c95265d480c29ce08d4c2ce59ff3dc1e56e8f053a958a75085890897b625de520bd6028bb512d89ff1391cf0e7dddfdbd160ccea5fc5b04a5ac03a7f890fff56d10dc01b9f85f00d8bc4710f35f29fa073a28f0dafa798e58a2913aec5f960d662222aed06c1eb11a216b2a952da2456"))
                    (acl2::hex-string-to-bytes! "2b05b170d4976409f23ce885a0a2c0a943226688d4f5bbaf35fabb46"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "d488bdda400932de56a9f105f0e74ee79c2ed869faaadc31665c2b5dd7691d7407b8d69cd5bbc80ee78d79c8933e8781006be9e2881177f361a2add47d59acadaecb81e70ec0aa9d9e40d64d2df1edd47ca050999f1d6208a64c47427d79ec1211ec4724b4bde071c3ea90fe86a788c657f596f40776a4567d80820fa070e58c7841afb98f5d70749eb22d4265096140db93425d87c545c72e66e4a3aee79f012af2410f6745648c76c1378094be7f2b4ea2dc3be9b24e420028f7666f85235b7028858e0420c3a3953f3622d96feb6df949edf6f3e4c6ae3deb43c1abf37880920b049e2e9496a04153b08ffdc4abf024b1fcf66e04b5f52bda4f1a29d8a65ad8c245a8ac49e46bc3252f2fc864c99a170291a221b924fe121aa3dab2ddb6cf8976d01247e9b5ec63de3a6a01f84c5519633464d6ac19b13cd6c76a0eaaafc9ef4e2e46c719120b68a65aa872273d0873fc6ea353859ff6f034443005e6992f5073e6aaf48301b01c"))
                    (acl2::hex-string-to-bytes! "e12c2c07628185fcc930d02b5fff6408e8eb1aad2bb0032830e04ff2"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "a67e008f13a29021d1493c7ddb0e7f4126a786f9e01991eff52c9b45d81d3d408094d0e7b1d5d2cd7c3a8a2b09e1ef3194a2c79eb6440c8fe6f093943fe5f009846e94a46111384af30a7f278bf8d6fdafd59fbc622b9054a68555c73e247d82b8f493edabbf8b413e9f619ae9b9fc4d8028587b6d38e4e63cd529045e43ad52260f17eb1845935963e378abac8e29c70d8439025ed2d2eeb3c7577325a84822d6a566c52f0c183b6cbd0e835bde241660cf897abd6020141042263e1dbc1699ce9fc9a4787ce28e601469497e10b6e081f3910a134c97ff5d30e08acee7de6018e688b07cfdb8fb48d2d2269e8ea7ce76d699dbe6fdff58a278cb82bf926a1ebc61eb10c2f8642a2bde914242bfbf874962e815c1db20c6e5022f4b7823807cc0f73aedd3d3acc21120eec4f20bc5b4027dc8df294ec1f230b656108104e9222e92bc6e35f8596c038031d436da83fca7117f61a8beac838a7c87f7639b6db70d19babc72cd33cf07e6a19c9794bf08420556c4c593a722fc3c0928a552d3e466737cfe12bc0194283a87419ca1427fda41953009357b778cde25cccf90e550408a4a5d9b2c8d626b642beece6ad40a78ad84ccfa0df029444a230b69bc69a3a3456c0197859de2da4cd5c9"))
                    (acl2::hex-string-to-bytes! "f6430298cd0e80d919f22cf1094988241f84b8753838afead0156940"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "012d3f35fe162ac6999ad84d9057fffb5cac15bc4e780917dd2530563fe682e8363c06032dbf8b8e831b993d2759a7fe65608e94ffbb3e81809cb02acd211c36fdd76b376a7def8c080a5286f6f1ccc953a399f55355ab8692049cf9a8b7e631fd055618dfd6be89d16eeb6e26a7d625cf7d1e2972d0b6eebc66db0cf7304aa0dfceafefb154604d864f4b3edf032eda81f7067683b4c980a5c957abe312ae4c4a3a5e040b7f3ea0ceb55c3d488e3ab010183725ae874b05f535f71e94e77cfee7524c9f13debe4a51a905b58acf636f25e99a7d1dd3a44656b3a3ac31361b7c5e1d5f08f1f94912b8322eeccc4e11de677017c8f51f3b086e49439ee288f38e56a38086ce5848c1ab59bf2b3794adc67fb7974c61f47a16e502c03cd1646cfe77ed9863d284211fd125a1baf3289def14c6005387d9c83764e300ce2352ee936a028de524043ef4d39ebae127924b21cfa33c238f568d3293f6ec9c03dae114d7eb4955ed60050858ccf3e8fc5b1d2458583104a2f332d019426cee9785ad4a65aeb2fb5421e02309b83516990a7dbf56edc940e8391dd99fcf690b07dabafdbd55b21e82ce360c8e6966b52627d2bc7168ea1e88afb715cca6c93c7c6271f487e730079eef6e8c8ee76f0bcd3b84e152d37ee5aade684b49ce76c91ea338f96a6d13ffd10cf26bb23b3961b4cdfbf7a1356cde784c9945bd4f56fa142778d20eee454c43ae2971a16cabfe03659ec505979c511783b0c9d132531687769a481c72ad361b5c7b650382ac2ce38bd7"))
                    (acl2::hex-string-to-bytes! "79e084af148e8073c5e86aab498eb1b7472e30d9093930bb6b819224"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "b418141ac57d6002e5d473c10797f137184f4be031fc935a12b78f21cc960c9ebdd07460c121a3a9a770f72ce2b0a6158104554a6226e6f0a209913257bd363d8294f18f3f83f0723fca81c378d76ff81b688f62c7f196aefc3d2db2d2574867ebabb331a669ed67952a3a0f32d6f4e9cdc522b0c9244c9c089b8d2ba42ce921e05fe1cd5e9c458f92d84a977674dcef7f6135bcd089afc1af6ad6243504c27549f6a4eccfe49ef21c1bd8e3ad78d0131986f10f45e66825d7f43326eda77a6a0eb923156108a1991c3d0c7c40ab775c847a7e308e041ed0697fcab834ddba0817f087d07cd2f58b7b840fe995d57ad2a68503a715ab7acb4e54c362726d2a584ce3f7a5880842d01f3909b216be807d68f1a8fd41a2231d6cbb47ac0f20716196d99c1ded733271f2a321602a2d3266b20497b979b3c2ceadfb7e1f73be382cdb1c9dbf13db6b75db0c807a38eb38946d08f56327a4ffe7f85e8f42060f26bf05f22314620895e4c1ba4e88aac1390f7f43ed83b23e816e342aa29d31dae424faa456a343c159c78ef6b213e774dde0d2fea4a28249f16e4ba8e1b95ae91963a4795df41ccf5f252908d4a257cfcd0e4d804547ff8df6ec2420721deb026e82e9bb618f56fb4e067a38ddbd6299b1a1d713f6fb91d239f2a7da7d514aee8b8bb3b089fd5a0bce7269cef5ec709930a25afddf45a0df6287bc1d2bdee1751785e9a2fec58fc78972de1143b8e998f318bd872aad947d18b44b33a34b165ebb7973d7e76f5cf9240dc14af0a8355ea20be8f9fcbbd57403803c620e2053731e6e39986d832b674a4cfac22345bcb3e483c29b80f8cb8cccf377f394e6ff7ae96c222cb88a63daa13571be0515e75f618935b4dbebde8c1b15e52e5c24813cc37c18ddef0c6b1ad1d30f89bea21c29c224298e"))
                    (acl2::hex-string-to-bytes! "662e1e83c3a4a506a649bfa6f6c6eedaf5b3b6fa05825d45c13ef2ee"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "51bcc9e9244e2fa1752c61e65aa5c592138447ef9287fdcb3b120bcfe825c50c308c14d7e9551bd7eb871a5c171192f29801aa3769f6d1e917d8a543c9c6138a7d50c0e394e7a867919106791ee27357502751f02be46af66981d53377401a20c17295b42feb207aefb3ece676feb0e9c15e06c3e1381119b6756447fb791ff782595bf3c148c9378e53954b274f91cf750f8f5792da0e7a881b1a265c5b6d364b6dda4753e8ecc7f5dbca4e3abb5d32f4a0b55646b1a94266ddbe79a2cd4b06a116da59ecb2d051acb1519ab62c5e39c47b406505a72a99d9221ae7d5b6008f0ebc22d0948ebffe3c3ff2d5a389da0d6f06f82ed8a3d8a66853f419fe00d5f819c77b555364fff62ceb77a809ba1ad57daba559c68efd929bd615b62592126a4b9fad90321cd960041fa6baf1a666ba74d10e8ed84edf84fa1445675054ff0c2aac8028389f95644d4249b4623706e4dedffc7415ccc6a50e754bd4892ee3074057bcc66ed37b2c8ebc263d6554417a423f37b385f0fc1d5e62f1a4d708088215062811daa1d3dd4ff406fe01e560a5295887f430a3c8137241fd8043cab4a62f228db286f1e4a21954b09e3ed3ab8c1a92b4df1fcb71afd37b380fd35c309b1814b3507aacf7a17243eec624046a589d55851b5951401b08796afa91a6c48858436294bdb85402e529861ebc61e5303a09c351e61e7d7c99a25be88591b7957516622556ff43eef76f414d0dd2643d7a163628cf1970fd0f5dd282b60353ecf9f77aad50680c8d327be7fc1f402ea6e4e5a6491b2960c3f11499702ce5deb26c4d74d1b0daf6061db640d2fb918c9401dcdb5bd23897f9c72e7659ef20a112547f163b803f4a8240881378a8afc62f57649c07fd2d5bc51a1332e5089ae4323bf72993c87e5128245a2a952e80249e70e6b3f5a2eff54f702a95bd7e3def3aa81c74808e7f502150924bd296eeed0ad4c8e6cccc68adc054dd668959f009532456c1abdc2a92944c9a3fd7e7e06b9a0974fe34807fca404b936ff0a19a7ba5360dd6bcb9675eaaa1d8ce4b293de4779950c3d7c8"))
                    (acl2::hex-string-to-bytes! "d2323cac073625fd6dd5d937cfa2d8bd01f75cb5afb15b7757dd8ac4"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "52f5a110dddc9f44f8a534eef9df0b22785706470cd4246c5604a8e041b778fc874bc3447867e6aff7ed9142c52925e3024f616df0243e00fbfb179bf6e30deca81faa6687d6c595c30ca5ae86d1696b7bb8a6158d8b53e2f13de2826620b0f5bb589f9f9290019e65ecb95bc63e9c5feb7c1e6969722abd40b451be18d727bfcae0ea1722fd6573e47314710cd8896a5a820e8e0e3f8c37f146ac77946e3bf3dc9ed3f5be85b8bc488b0017e107f732d1b2c1af045c41f80add3ad4634fb439fa7aa262ef90c120991bc375cc33ad5f57f47ef2d12f11a9966044fe0ce05a382b5c9e853c864948e00aa6b455c4749af5308c7fff22c4f29ab7121d178bf599772c10872dcf18dc2542f1fc0422148bb24e347f169b217a20caec580db74c03e6d2408f4ad3f6836eaaec7462b555d99636be1a9c04f1ea2f811ad775cb281f813776c61cfbabe5c52adf45f6e6ff2ca9d172771bc4fce9d0c4e3c3d73151954f41fa532aadfeab31aa1196481bd92ef40597bf1a430a200a7b9144bb508f0c58d9135ffc52b97dbb12cb90e1bdac857ebb771922e9e337e58e6a81aa90aa2da5e80b121b53a202694a555b483540d94fdc58aeba1e4ed569e5b7a32c59699f3d5428930e1682e3453bf71e05ea5dd3704e83a66a4a15a29f68bfd1c713950a2907b30b5d0520c857cbf902a143cb1ae8fb4028bad5fe488e00cc340526f6186fe7f3a191874657a6e3a22ea3aa5157bfc9d75c3327d26c3e9324f46dfb546f19de327e3103ba3b3a6c3bca9386b932fea28f6b5ca858b9b80a103887f877b72ed3b6c9c5e3451a1b4c0c00fa3ec952ac6db17dbdc15ad4034ed828dc862799b7adc9abd68eaf9d526d5df65f3c4a60d11285b9dfd89e682746c4edf30e2bbcbb56b14ceb6cc06be5b576d44ae70303767853484957ee1b2dd91caf54f64cca42b5b6d38ce5402ddf0e9ec343a7bee4a189635283a410951cf1279588a7526855b222694161f68a455f1febd3e441ab5d2cfcfe1629cc7ffc752ec8168a0540fa353b359348ab2b21903f73810655ffaf496874894cec81d64c797086062cfa3b264b60090227b77a884783d675c288a1c9eddd64773642b16050b36182913e206d98112cf732d759317f45599a50ef3a92d4613011746b9598fbbc2e68fab2b0f36b7182909433c5722edf92dbc7dc828d652774826ca3"))
                    (acl2::hex-string-to-bytes! "bff3139ac7bf14524d89645284c3541679832b6d4858b3c47ebff434"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "e563453d6ad52e08bb112f1e6b6c208ca47815a1c91d27a25aec78bf18af71466a454e1700addd58b6608269bbe47f1af29e721c98e81fb6286370b97b8ebab63498426e5b73008a482aa38dc374d5066989bf0a5234d0e1f02aba003a9abe370a052a2cc450200839189870cb45a8b7d1e204523d2cc18c7ce85e7b9dfa0432b6340d38836cc43951d16bfc1a7f8395f25047dca1f57a1d92466211ecd7150dfeccf37ae0ee748557fe2417fa49eb08937c9290c8ba83fefbecd0ac6e587732ab76e5da23feb8541546c9abb1cb6a787e0dec56f9a165957f81af336ca6b40785d9e94093c6190e5152649f882e874d79ac5e167bd2a74ce5ae088d2ee854f6539e0a94796b1e1bd4c9fcdbc79acbef4d01eeb89776d18af71ae2a4fc47dd66df6c4dbe1d1850e466549a47b636bcc7c2b3a62495b56bb67b6d455f1eebd9bfefecbca6c7f335cfce9b45cb9d2be43b9f6b8e42bdb6fe3a3a513aa6dbfbcf684cc55e35bf245745d645054fc3a0cbb4aed0adfb56a2117dfc6cc7617da048cbbea03b514cd43eaf44fbe2bb71bef79bb58b0b35aaa02de78866f7784a01cb4f0c255ce3b82b7073e164e5f7efadf2567d43b0d9ffd9d9e0a8e8cfd48b47e05c00fbc8d4495077fb752915eb8594444fcea1a5d09e759dd0229a595b87990f5a60cb870a486cdbadc8f17b807b9b73ec6f4df100cf9b011fbf239cf68820e1b1f53ca26ffda622f3a1651d9c18ec09972aee9c53da738e35472ca08ed403ddca87f720eaaf27d634ca2c0528a5ac8eb7c52cafb00626cb7b825c6e09b46b8c0bdea30962a7167ed709ef3b0d6a0688b309b131fe9cdc4157385c0d51708074ddc1e966c12e9ca3462756c531af2081f5be222a27e4bc763230d3854c50691fa9f9713eee8ae1cce8ec08ad7ebc2200e4ac2aa21226eff475336adfe49c4f15d54fe299d1030c81251faefbdbd8358ee89a6fa2a7aa9d438609c922009b59dccb88e57f52e3464b658aefb52f4d81b269529931ceb7ece9cf7a97568796b0fef16f21f95970b6b010f374e564612df4cf9a51586f970770b64e7c8363d57b03b1cb3f0ecef4853a8e23e592615d61dc833b3f1e72901460a54c22b27ba0e3717263e6c799b5da7148e2e9c50e8947f3ebfd0a8bbfeb3c3f033d2eebb73f10d29b00bf2aa2fcdedca4f72885532c04c070a75aa73ff0a69738e94b34fff929ad966a7142d210b8030b2c4bda188fc3cb7a03d97bd8e67befac181f973b18578bca69a5140f3b916ee647facc45fbaa9d72d72d0e7317e2d0c792521b53e2c1f39fc503aa42c391a69b11dc64c2b6e1bdbeecfe68f9c904c1f11e8dc8"))
                    (acl2::hex-string-to-bytes! "08a9d2171e9c2ff730050bff7fa4ef91e4b004bfa06a452a13d6e087"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4e5132de5ad5300a472df3132e85ebaa94dd7dd0d910d390900adaff"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "c0b461ad07ae4ab2f42e33f85e7fceba9d6c4db29c0019f13090c70ea8fc4882a6315f532df258f0fad9f85c1898b08faf85151a1d5695094728620d31c7b3dbcd292ccf113b9ad54ce1e489ba373f9d561ea07beff4772c7ea29a79b7dc7f50dea4af69dc107728924ee1c66de5e3ac3b3dd51f3d6228227d05569861c415aa81abd068b6fa57b37328d65a5929048305ed9c4d2d2911b3eb597eb45806dbb05d447716c680a23c9878aa8ea58b0aeb5d33dd4ce7d099a771a7f913c42a3b2c1b7a8cb9352ac18ca94647aa7a935daceb19806594df27c5b6f6b1fdccb89ce46252f79a2a801a6b4d9894f6ffca658e56a66ee05cde25d6634e8dbc983966a2dbd6a001ce0b16a12c7879b95db78a5947e28d056656faa21fb775975135002c2e0d703549521df5efedde3dbbe335d465d6c61bdbcf0ef1933a51d51f32d44a02ff4d80c2dd98303effabc842d4ccc85989026ce2cfc98b66de09bf29ba49c6ffb520286135ed1ac0b7c8a31914b9fdca03273ea4e0933841ac869d9854c767ce36d27650e1867b1e1c1e166a2b623b614d1eda4989d4bfccaaaeb79389b6f4178a49b2b469c91b003793685a9f314ac66ed088f853f4e9aee26a5443df7f3681ba6f24b85620c20c0e10e313fa8cf1a06e2490d9c45b7e905ca3bdd0e088f5b5cd2b14812ff237edac7ecdbdc3ddf99756013915667146191f3f3b61143af85bcb64c93a7ca7434916c36f7f838b2b542d51ccd4c19f08f23328141e02fd091abd048c09c6a7aec956c91182c1fe4d7ffbfb913d5b4b8ec2a0449030323aee9aaaf2a516935940147276fbee059e27c0c9e7d079bedf9e326e7df9bf2c5770849a165c8a6e27c0d375b474ead8d383bd9efdd1af72c101eaa6b995f279be57809558f88ddc78f8ad646ac59ec00527218225db08fd5b7f9150e2aef2a0d56b35a4a0c34667b5ae4b49803733479cdc927d4738ace913dfcf72df8ae72906744cced603de9644ab542bdd10d1f5006ab8984b2ec2be53eb4e169871f89db8396c1ce2017df18eb771136b989f5199993dda190673028269b343f5ad4d8252833e94ac054901fb994a081119c6e3d031c64c9908e63f0722a902de850f633e1a6a411d576d4df41b8612b7d96e330bf9fef4b47b44fc4cded65a3b4066c5e3c697281b3e0f5edf03cd394521bdcab3ba8adc50a9b4c2b864e253f4277932a94f6d538f0458ba251ac986a439de2f313c150c56d85f18b37e76f5f1827131f87c96c962ef6feb2f282000ba69f7eaeb47458aaf516c86a5f23a491b99754b3816b7927a487c7964f57ab070f931844ba3c260415c0494f689d55658b3ae06d1e57bb7c156a9ea2b48809caf30b384b8173596f6404fcb0f0bf024911944ed2cb1873f1e631cc8eeebd9ce5d58a6729a6d4632b54774b2564032bcaf145e85af42863278b5fd9a14a42e4e63b731d7586b9c3e65b79cd234f0442598ea2458c3601e98b99575e9053c2722aa957d395cdae0c1f0d9eeb112e2cd6736e3bfedd1d2f77ac84bac3c287b6b54193cb1230d81b9c43df593d956df768b6f4957cc3f61badf8d1fc91be728a3ec97f3f61ad32ad04e3453b2d26aafe7982d62137e76d50f"))
                    (acl2::hex-string-to-bytes! "86b34d3d4bde0f05f9d5c409c0ff134dbe71d2c41640e081a9d3462a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "197136a53a6bdd29e64182d93435be4427f06ba6a14675e3cf886b57"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "d1c82cd0224a5a141643726388c004461dd8b3aa540d4779ab7f0321"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "f97cf08fe663a2506c5c8ff6284bdd9edb71181e9271d3fe19d4d205"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "05be5210d70c1d8de9c777a2d7c5a3957a639c53015c1020ec0151a6"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "12bfb13c57b24ec9088ed5f6fe67ea38010d00f6744cee29629061e4"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "d07a5cfa89951f608bac73886f5bd9f9c5bf4fe49e9b0ac7d307abfb"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4d6d3f7117c907aefb7bfaf2fc0f01960c4968e824fe0483ef66916d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a44335bfa12a61f8971109350e6ffc62c1237f3f662b7ec27b520370"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "d3b2c5168685d9419893d8bbad9c144bf4ad7b416cd5bd2c2d90a11b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "9ed5ae504ccd18400fbb45cc91cc19c72fb530ccb323d1c945a94bcd"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "491242cb9704f4776c51e53268c93bd4c3e6ddbdcf6e153091fb64dc"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "f33b1abef61613d17ecba21d6a92116261003daea6ed288a456ad012"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "450587633e9d302f910baae17d3704c5c570a4990b894df1bcbf92c6"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "729a8e1c242642376db87cf530bfc351c82f0511361cdbf87e870f3e4e785c3de14d746d235148921690833474ea575096677d4d94ca7731ae691915ea3b47d7ff33000e72cc9c43ffe6515e3ad0fc5e76a61e4de6dd2573021ceecb2c4d87b394f539e759c0c6b0f161ef59ca9839e33194f1fa747c6e27434996465cd6a43b683a28e11f554e43b3a9640f6c6b97478fdafd3a85d0d9b339971a70c2fd24d542abd3e20eb2bd630f67b86668719df258204bf66201ee80acaf812fa6eb91ebebef44d36c009b77cb43f44805d1d34d540de117543635f66438d36d4fc872e42caf839d82666ee9b0e41f9c8cb4c1be4b4ac983a197377d28db181106bb5028cb939626980dc02086f3414a4541f0ae047f0b24914d7b7ec240590313ecb21f90428758f61d18841517633978a1e7864853ade06e80895ff80e9e2bbe84203145655e1ae3bdff7d24f98b536dfb2eacdf6eb9a17e57ea8a93eded714fab22c6b0a32153a915fc737d40c47f6c784e8a9179364e0bfb25655b8750bd0d8841469ba5daed8e4e97cee69513a5bc40ce86261c2dcf5b94c25a1781cef741440419cc1390cfdb9058f628a082aea67a360a65fea50cb6494ad091866f073306f4a70b944ce1d702c8540b081880724f0f7d17c8a6ec6642442f8f3c3cd52bf7975822d0150bcccd6e4e5676dd0468e6df470421395ab2a838deda4b3c49713f7895d770f3350018d24dd391da22cf6831d0f4b2643be1a3ae473a5c784c4c051cc4f41cee47bc0c7dba480216c3f1fc8a7faf2331783c5a2fd17617f6c406710d4f3f7b0f2948d1f6fb7e7099cd14cb74befc1ba8f90bf642e45823c09be5add29c81365dc4ec8ec57a3d846c2a81fc76f5a542d0a6969e70e8a629deb732def31581f4e0fef06c739d4ce696c080b8198a7041b167247b404529d05602342ff820095ea1c09b9287ce753d1f795827e3e626c05325feb53a64a9c1326235c66a27df1ca7f3372889747e31103b53705e718f66bfc66d73e6abeea37ac17bcbd191ccd5fb7d0a5802f6306c70f0b7bc129372dcfc34b7ddcd83bf1f34064f67b3c69831a3485d4879be5393477fcd3286b598ca1255241c7655e5dd6a19f9c19a2e0bdaa93a77207633af519ed0ddfb423dcad60a13fe586987298423e2c1030d31ed75143da20981bd0b4b7ccb2f3ece95ceabeb558bb82f8351bc6848ef52953705aded38242b23c4312ce9580a24b3236cefbd0215654faf252f4036cad9b9a81c273b1246ade3b0e79aed5fb8d578516e4b8804e28751756f83b3919bbd01ecfa8e06b550b74aeaf63d91c87f78a5869a6655423a45be3cf560c942aa29f56b678857088a73dfd826bfdc7ba835c229bae92ba467c71d6b806f8b313f431e8d2bbd35f33b072ae33ec5d7607a3558a4947b1312b036f66cd8570046a23303bb0dd7a85ead9bb37308fa88603fa0ee5ef1c514745cbad2a6c5c1306a1a222c5bd77bb831265803f14a03f9613d9e0dbf986e36b2b47a37810f552483f13822d74f536e54ef65a1c3cd8b324cbe553c2f531f81b5009fb0657dbae37a6d9c6c953e07d1b050eb15535938390a08f00316e880fe638850c2bad9997538d77792ff6b3fe92b9012e808b6b218826395eadc2038b4a593b6f7e8e999e3a477f01d7ba0a2bfaef901a149aba2d79b196d7ae9c3f0ac646af8d657766334a594309473a3d85a22d1e1c88001b07c3f719f3aa6994fec865640e770479e7025775e7f4e3d8ac36d61d9e51480831155b253b37969fe7ef49db3b39926f3a00b69a3677436600079f112816df0c49dc425b36bcbfda46997877bc13bd90acb3fb027d24d56011b2cc9b423f3191a297612e4ec051beab6565ec099bf298af9a894114b4165161b7030e6540ec32ebc52c111be8d8c786c1868a9c48bde1c32305f8957b82a46092c70d09b26b5a85e40796067cd18f593c7ef5fc909c0677fbe57b9801a6e3aec48cf9b80a3b82f172883f9712b860a344a390a30a09614e73bc6c74dcff999ccab966f8bed946bf0f227056411a05e56898ed082b72aca738df2db463f930cb23f0c73456c1105f087adacc6e62c9a016c38dc46b842322a9d241ef4d2043ba4eb7bd03588f6a520cd5d152112f3653ddb7e54040aa7f31453a3b7def113cdad9729ade4aa0df286ad7f69fcfdbd6813bea761451521b63997b9cf8d028a6bd22b5f33d8381c952c0f5acfe00f524a8afdfc2a8f417823a252ac8e876f2fbc1d90658a9b9cec351adc377054b079b4ab80c5d86df673e1566ac51b569ecb6995ac83d9b53c457b57fa5e74e048fc4e7d2f4a92649cdb995dc94c9f81f9251942680148ac82ef0feb5317020c3fa1bf08f3d281fe9b78f2c055894191e607752dad26010ad5b198deceaa4261fd31b054b2a6ccc25b129586f8c32a0dcf37245d613c29569c54bf38525e5d66ea73c86a560f9d38457129b79944817d1b7ecfda6d89eb1941a12b02c97d7ba46e31b051c55319e9bf5c575c20526f6a32a41eadba566dda2b5d84441a4dc3a1f2310f11e9ba6a5c329dc77b74361c5b7c71b8dd2fead721c25374042d175ee7dc3c8eb3a88eb81907fa201f7f38afd861ad4672b38f52776e4aa14cea2274fe961ad53dec152aa9a43c0b4d289f5169491f6f75308b06629709b32339b82c6c86938bba9c99e993732b2f0f57b529979c90dced8d271a6ca5595392021faebb166f582fa0601c5fda8e356cf4c060e9c2c835e2b2abb61d0002299f2da02372ad75e6a6e0b2ec96432ec46de8091de8238f215d875f04fc10ccdc55a283b31ab26eda2643d981bf15d4851819d8891a27ad4864f38a29913e7f15b94910853a8f23dfb8b31c0262b8461f777075cc0937e95302e1765064f0e27bd50a8fc25aab599b6318c29808944d3b92d5270d3db90cea3c36699ffe8fbeb7b0ce4791036edbb62f58146a3811b0816f3137743d17deb41cdbac3ee684db70527b811bbb4070d9245c5aa17a973998186aed2a91b7275aa9b88756a436014fc4fb85870d89dfac4b0cdc64568eae29956a9157c103af83ebb5410fc10cf8fb0d349bfe7eb1f1d6ab50e1dc555ed8dbec176b2255613ed8ebbb2490c470871d35100ba898206e004c11b330fe6f7f57275bc3b4d63b9b01b0b0760235c9785d45761cace23f1ed137359889d9296b454e1de035ccf88fa5427e808c9d7e9162444d1733e474c8f273a6d90f6ff5634b67450bccbd00e26caa0b8a5782f99bd53a7f44642728360a084abde9473930147cba8ca133bb5c00ef41dc8b6fa15d5d2ff3eb6832c42ec2ff0b8e40ea5e9a469466abddb591c4061906bd4fe9dea1a32c9154668dae79c25fd0923bdecd698ab830b0a8cceae429314477cb7a182d4f3841c787c150dfb130801303aded6587ef00e229fe2e820f3ee25028a9b7ca085ceb8bf3551127377b0557bfde2556e641be8a89d6c5676048c8918efe2340769f77e9d1d1581bfd4746a798fadf034eb652f0efd9175bc558de812402c5574209cd2949be8f81ab9c2b4ab68882356857a55c84918c26e5740c6a133c03d714a2"))
                    (acl2::hex-string-to-bytes! "1fff3932e74bc6a0ef9f7e994889bf7cb3af3b892af186983f4b0b1e"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "3bce1c957e773813765d44d218a7c14dc9d03a57489ad36331b5c90b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "40a3de5449297067bc4bce3068abf627d229543f34ac71fe9710d7cf"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e4c064eeafe8323b68aaad4aa6fbed8029771b7644d49849d2cbe96a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "bc824308b8955420c4eafec6a9948bb5d35ae2f027f5813a063f3927"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "dbf37238c302735e88483a64a3eeef0c626defd048fa39bbbfedf112"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "2eb0eca524d5a152f51af240c51bd815f2d52bd9afbb64aeee06a7bb844c016a8754eed3fb291e78ae6d783261f0606b2881358e14041ebc0ec507c1060ce6464ef243474072f7c827c334c45de5c567b07b977d0ad87929525c20da8d016e7b4931f4ea3bdbb5ac27c52cdf0024916f516a598ddb5ffc5d7f73faef5f52207141ba07a33489df8b751b6a6993d24bedd2f681a8cdc02fc160b19ee9c5288490fd34875972a7fdccda6cf64c5e1a7c48fcc819b52a249abbbcbf13ae07ab2341474879f63e5b97d91e60d66ea922abe89788289b38b2c6b45a4461286d0284eae59ee761e0cd361909a014b9bc1260356bdeb74440933a1503e9913d3f5630f5a0ddd882a02761f462d4b813de1324973f70cd9789ee7e447f56aad7c4ceadc840ad5541c58ea1d2f3a4c866594d7d76dce9727c5dec1bab9632b4dd459385cd8db86586992c7d1b82b64e9973e7f63a0befaedb02b598b01026b8370acc3f6a4fa3103962297d37d8b551947509144b922a8a9cc1707dbdb4d9bccd5ba264094d195e4269ad298fbd47da3fa7dd611139440da8e24553394bf4cb87803d7b1150c05889f6df0a89f914f082508864f5cae240e5ad5da08c9d1eea4dc0c40cea204005f751e7a78e53c59b45b935a55d0baf293c6ae0ef1d720023076560a0fb85426e72c1ee88079402e3702b0aadb2c650d29560f6c55692417d9db538beae476e99e38f1c365a8472d1ebec2f035dc17043d9d8c5ac9ec58a01f7f975ee6a2d6b9033049838f1ebbe6c9e44cd2fc4c5b583ee1a782b5dd283b5fd783eb89800e25b48668b1632918ebd717d0cba5199c799be6e35cbfc82d3a3987cbca59e887a000cfd1c59873e88ca4dde132b6b074f90301c2412e19d23d3ae4b163ce58ddc896613748857b9579d18ae0b0a79d5f56d2b6b6f8a75eb1a4b282e7505d5d7affd3deeba2b09cba13ebc361be0e5cc603a99a53bbb20336115ef6d0189d62d9c657bd717cb073c0fb31fbe6c3316b439ff6110b32c4b3f9f39ea50e14f51594bc80716b3efb8d448062b8d268e745527c26bf4fc655cd57dbd001f3ee0ac21632ec9f3b51f845af53fc6ed70766776d38d9f9b41eebdbea3ab1b213fc2514e8da0a9b645e3d398bd64da24727557e1e467e54ca850d4c2b80a229f0d7fa6a870248fa68dc40d54497f4e7b5512ac5c0359254e59120ba865e3fce1cf9f15f6b5c5630cb2499a5740970a3a41d6ceb0d86947ea7dbcc93a513d932faea8050203880d30dffbe3694bb5f6a95d4726a50aa133683cd6753e2420ced1d224de920377755a3a604e5112e0fa10d2ef1a024209d110bde9797d8895950e629a076f5c9984d2eb0244d93c20feb6c756139127a80a29297c3cbf23a2d0308793ea61b7f620df1b692ec7727deae465027a186286c4fda71826cb9d52677d6896ee2fbc9732cbf21b72e45b00a6e2d47db5609aabe8d260449f6b9faa433e2e2b7ba31283a3dc97a27af8a0b01af83b4f2e3178205ff17c880f65d1b44aba00daf98362c2595b3386accbe8962f6ec2fe4b52a4ec0fa2226cde82da77653b072cd098535d3e90ed4d7224dcb8cb8b9314768dc517e22d7a79748a63fb7d076aa869aa8e5b39c2ee8133cd2320d798d31916a883aabfd9ad363a451349786752d72a81bac4313a5da9568c66d63cddad93926c88ac8defe32c967b936840a3e73f7ce480a11041634e452a4de0e0a2470c54a1792536b743e1f1f38a60bb03506ac89c88f9bd1d28f7676f7f48ad2fc90bc30f011da179f6dfb476a13ad3cfb6e0747279d42016d32c2422a5168acb5c19477dc69db67d24ae45cc80fd6f4e4027ea2bfc3a21768d76e40d66505ec028bd48b570ac5d0af3d40ebdceb176c917c99092b834a8837d5b0ff66b001354547f2e7da8537e6a8edb7511335ed288624ed06e56c99d87cc650da65c2d90d008e7629e169901e31b4be48f590d6ad34a3c11867d02e8c8a11c47e5e0e0bd7c6e7e4ae06186cebec3aac200e5c626f2f71dbaa3d6e5718f2507237a9f20829fabcb4f0a3122ce43ad82772c828fee2c9078358d087d21237fae0a2f42f5c782ea56d9e63076e7b9025a6bdbd672aacb3ed6e90cbccfdeced89965ca7167746e51a8513a0489026ae9bb06d58976421528be3ee89ae591d658519a14d37f71a908481dcf5563089c8b2cb0e144a39e5377b89ce849c62384265d13d51144a67b38bea76dcc0ddf980f35d1c9af02a7cc524c6bd6727a7a206b4536d5d2bffc2e099356db108e71a69e5ed153719a43cae2031378bc0a80f24e926ddbbff65909a24e91e3caeea40d6030184832745b934e64fc96d07d1b41f8cb1fb180f3a7532c77c24ba23662177c320e6f9c96d870bff88ffa55db35579178b31dbcbecc55ce9750dc3f6d920a1bed1b92550aa0a95d45326bf42cb57a402193e81b5db257b135a35e4a9b1c2f64e7a87705d89d864fd54eb0557983f32d4f26da47da3fa0fcd923996ac8e7ce80aa1be75506b54fbe304c921fc9870a8189d4bae53c5d7ad80b385548fddd2f96bedeaaec9ed24bd1dc92d9fb25a90664580552a08cf32b6d544a598c42e599cfa9578874a7dbc3b1638a4d9094d131fd4321681dd1e8fce23bc16edd9de75653f5f049922ab8ff854d9bb5871ee7d175d290c5071326356049055ce318ccac492910b4f02e514776202cdb87cbd32ff38dbb31045b941e0e047eeb47ba7ae4d9ce1e9336bb8ee3ac66a78539a47d93a0fd0873bcb01aff15f694e2b18bda452d6b84ac0cd9571a4ea6737c44b0cd44a4a9e055d91aa01f9ca7b00bc894d4196a88a9c7189194368569f88852f8ac8c2619ac9d00a10ec7cc9e382c35dc51a6bbaef5ae05ea2d7d78369e70088200b14ed6d214db4e4c3fcd63960ea06b37364663d02879ed0a385095cdfe1058b3fc4567bc82122843cfc67e3079af4a700f3332a2fc1ca113ae340eedf81282abb53354d0af4d37d9bfdf7870350d80e279aba0d3f3980e31294d2a6627b506d224402678623384f75642d7bd09601577bed637cbd5923c4aefc270a053a2ee0e9192641b656c1686dd14267ebe51121525db7ef36ab5344749bc690444814249435496207ea1f66c9c80ea1d1c877ea6f5e2fdb179a6fd117f639bf8c1f9086a83ac45f1767a2a93dbabad01ebab611177153d6ecc109f8ed9de90e594b24a5999c2f8f5149ebce1e2c0f692f364c43ef39ac593299923a89f56d03483550ffecdbf379422b6b340f8e54e38fc5bae15b0991d8659e9904618f2e14b578a604166cb8a366669189767423c5bcd37277ef83bceb124a9ee508ddb4d94d7f8440aae0d7a43aa60f2cd750148632fa1376111a75f7b518145ad46389c75cd48c341ebae8e23b44249cc6b47fab8ea41e7a64406584baa189e478b325458ff21e5ce66d54f3f4cb8267b913fdbc2c6be3e35d980428bf5170005a4f1ffa4372cb39967ab43fd3c6a90c22e0e72ea8bc144f6183ca4a2fb2574cfd89c24c13b11348a3658f4c64bff8f898bdf479600f102d2496306f884eb018d715c4c17d377708a1a03ea0340202bb8f1d8e32feef77fafb31cf0f55811af27b558fe0ec906707f5dd5ab1ae4746d38a621cd1e1d7ac4779f90eb013e3e8686edb9c20ee4673b5345f5c175ffa45ceff2e38dca5523d71055e77e84ff1a8f9060a6207feb42d9ec1b8bb47a595700c61f46ac9a5e9ac2641d4832e6634ed12e5848a1c9a73e533426c10a93b93662a57471b1f8ff868d593fdb8e8f328254491bc6cc54e004b011b0bd4d27e8e88975388d244b028f3ed902ca1ae7345d21c7f2492a885d3cf86dffa7049daf167d1cb4c18efa2d5edfcbb03a5e5731883f9225f941cee88d342f5ca8b85fd46847843036766f499b77168f55e345f0abb690fcfd9c8c479f6b3f0fe8f0d41f85ee93f31364111dee0fa4e843301a3a6d8fabbf7ec02b431ba585608e34645a167a334781e04873fdae5c022ccd3fd776d1c9b5a33714a4a9eae897c20c774bc2f44b72a8737c11735c262737431a8f6f84d4279588ae32f04353fbae0175bc3263c3bdafa415c0f621ea9d8c19152a00b1b35d7b3b60b599b7abd5779ac7730ecf8370bffc7b482e0690ad54d7976fd05f4e9f7ed894f44b4fd96d4860d338fd4172d3ff021771c512019fb706072788968d35c6e8b1b8af12c50b47da02d5a7bf7a2e983e4f03604e23bbdc2dcf1acf5e58667af31fc96d0857f79c90bc10b55482c8b1388cc5aac0df49ecd456104ec0977cc37f45ef2c397878700a6ab3cf0573ea7379378048721e900e8304d1adc332bdc8168df294ae0ed8ebd172849cdc33dc0688b6fce56cc6d96bc3412493701172b23f5e994e6e025d346c3e4cc5e0abe691ed730127b4c06c2a1b75d3a8cd1de2"))
                    (acl2::hex-string-to-bytes! "4784fc2495fd5a20178c7a8aa50686679138f54d4b9cba149269f191"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "735f3f0ec0f653b00db6fdd424eec183433cd77461b21c2ebb8a42a0"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "c034c032abcff2da1e21f1c210355b929c6a9fd188232721aa13078a"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "5e82f3d3fe344c0b4c05023e58f9f81efd31c8caed10d535fb800867"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "4400a4675f4a2f3ea0ba303ecc354da665cae6fa1a30dfe1556d5e39"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1878f939aeb1f29193f9b1b6fa4da7431de33404e3027537df537554"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "5a8aa485c316e9403aff859fbbdf00219e3b15a21cfe59750f261db2f9be94d1480c3978d80b6b92ba1c5aa134f8275508e9475df93a297d8ceb18e10caa62b1d0ffbae5c1c7db5b16e36e2662312e2ad1bb4ae6ae2f0a902990b667582ab4457bcbf073843081c2d7be89d74e735cc7801be5e443c746d66f7bbfede527d1b5d0d8a3cb2b30a4f08414725cffa4fb549d70e2ffbee82476e515fb1abc87022a588f9a95bdae8d0ca54cf25c7dad51fbb12207e0297ceb9814c63197e4348fcc99569f784391853fc2d0d180d177a7e243991d2f8b6ae46682f72b0b6514c0c21d96781cc4c3d2b16407d76f02e94431db3ceb93ed9a5705b90eaf1f4811dc356cb8ee2a482bbc891259cbd118ea721101753bb556f48ba04175867b94122d4c0c22c94ad8b18df0552e77501c6477aa3eacd33dce60ca5da09278c651fef930873ed1237a3d8b0db4eea192153e9c2319477cd25ec65a6460cf4ca887bf89086573ce5e38b70009b701123f0b6029a41baffe1c0c0e84de0893cccb3f3c25fd2c2887a21a07320be6d0920006b26657ca4c21c5e22090f32145bed4815d153766c7c59827166a3ccac1e97d0f1b9a62b186ad72e8e4c6ca324feb6e530aa0dd9ab750e867caefc3ea3dde5d2d860fed55c669cd25436e8cc3181b3fbf2b0f175160f039e20c0bc9cead3a4869d80b2140721e7ea123cf605d0815efcca8aa0f122a6d33a1c50476ff77c5e8cafad725f9f4942d646e326f8787a3278269b9ea657d8e7fabdb23f1040547e258d4bb2208b29381770adfe1c9fd8b8fea5784d3d2c11cc50838b06365bd88e11b4efe40f616e322e1e11eddabd296ce0b5853f0e25ee0acc1c2f3f3ab094f8b1c51eaa649864bb9aad9378bb66e8c8dee556d691cbc9fdddd6333ca5d50668862c3c57d8c6c95eedbdc7114713a357399371b67094324006402ba0f03f99381f58ed8ec442f87687ad116a0649c592916e4a5122f5c36bc5f86ceeb266e9f1ede986b1cfb37ead0929f6c5618fdbf566d465391186d9cb9d5d9b8864d16520624cbc63538ec85cd2830bc5f14c630a3a4446c0d4ce01fa0cee2c007005973e2cf0d305e84b62208466b4de3e05d74b7c5c53aad511b871952892d97c79fcc59e848f3bf66fd8728f3386909052c533cd7b3039ab1889ccb6ab6495d2b5023147563558166297dd6387f56f90a9995d7e3f510fbf48d255f1d515a82b3d441ba81469fcc0ca2ccba1c285ddc770e0132f073a9f9a094ec7c7442a92de894eef411d918c7e1d03c7ba71cc6dad5f260f5061516bc3f04b7cb75717e8750f6f16a2f040a9b6c1c95c2e141d91aabad7ac82dc39e6b58454ff42ac721b8756314c3e2afbb6f35ec9e36d380dec581a717516e658a2e74bf7c9e9b2ad9e7df441da833c2746cc5a9208084c21c02e047ff3bd0e3222f1ada0571cf83ef3c14ffd5c028257dbaa3bce447fceddcfa4ef7f92c5c25ae93d2feb417cebbabe09efe04b4ccc01e80578baf3444ea66d4551fcfcc1913768525934f35a0b855fb05a7722b247b9fe76a804cef63a202b97d2547c50f96c22b99019f764e6435b4cf64e6ce3a3f47166302811f61fcf8d14d722abc4694d9adb8b06c7fdea69334869c2a4515f26035ca78186c47df8f434d6e3f5cc7f604340a087efb5b65b6a3b1b60437bd7e306e284cd7801c3510380581504a99a58be816729c90ca8be728d91c3d4c3bc900e704e58700e8e5c3d7fe142331c488b6f0645490e635ff6eeb85359d2d96bd17c8e6abf097b3d7f049bf8d82b75186cc747c1720a6a3a993153d69eb82d641bf3a5606d9eb86f3dfb5d05633844789179ee72415781d719ece8a20f1f2043c3e65dd5e0ba46821d844bad5fe41ae78dd942ce23603aa3fd63df2c275262698dd70021b4f6b6249221e0474ad968e3a677ed3b84df052fb848bffb740c6473ce4834ea56be87b53066a369a062af0b9fd85641020e61d8df0df39681a796c109bc8486527c2bb27cc51d9aed3a6dbdce7d16851798e14dd6262273bcae79ffc496478c14da12aadbcfa741adab1588b6c8967e21e903ed4223c51991ab9e7e838826622a32fd1ce1d8d8f45e4f46eaed5d5e72dd793a45877943febd0974dbcae489bf1f311b7703934b297b817252e6ade72996e3f4c52ab927a97b528d760f4de9e9403f2e47c5615667820c185f173cc8f8ea93e40ab0460890791158fbc526df7759b7861dbbd583250e29a97e038c7c77452358849cb23f4a12c057db90f1d2ffa854b3573b6269f87c74b62e301f6a0b42fcf2a16470644cf4da1a83d7515cd8d8814682e88a73485977ceee31581a4b409b230f9c22e00f8be911831ca7e59956b7b03f37d0279b31ad444c6bd915bdda786a1d982c6b083457cd57e577f5b7af7270da9db60f67717fc6b76414e4c97d3f486a0d0f4dcdaab0be80a64415ca3f4c18e3aebe8343a589c6bd9e9dfff4b53a3e635f4e00d64b7e558bf39ece47a2af08f8df5c3ba65a35b4dd9fc6cfc5ff03dc98036b4ca8933b22d841da77966605d168a4118105ec1e1185dc41fcc3d72673869de959cb1e51c6f405a9aa5cf60b02e191610bf8f7ad7d44c871ef64ba1cb39d833ad0a4497c927dd36eeef1df090dc22bed34e5e95e58a908e9102f34511d7b57e249caa0f615d42facbdfb9d9a52a8d7f884535dcb2411021a1789c25e117e80722da4c168d1757ed58fe09e2981f42a2572d7b15e701b48875e1ad0ae1035916ce4d9073f55f0459bed376d511c32a6e698c98092bd2640a37c590bcbca2f1f3507cdff0403c318424e76979b71f2b2de6edf48027bd08f00caaac0318fc96e6d1efa25316b7c6b7b3c04895333a2e25f6d5f74283c988831e1a69a79e1a3136792f3f6521d6ab3a93e87cf22c1d1ca0bfce0916e6ffcc08959c2f6ff503ab2c6fc320ee06a3c955ebe76e584bcedbe46c5f92c5f865bdca99b96624c5bfea10dc33abb670260650e343858e385755392311fa186206411050f7768eb857d82ae7084a8405664b72a586285e9abf70e90c6aa9d4cce74121e5e1c87574d25cc1b9c964216d6b81055637c18e1a0f163b51f407c4dd12a6467efcc2a1a1385d1e79fe6056513d8e4f21dfb7e75f4e2e7617377daf3e8e56c744d2cddae892c58a5030a8e959e55c992c0572311284a0e389ba715c97633bec5eea52e7f1e77b4b66e6a3a22c6ad12a9f315a34daba54cacc054c948f9a3cfdf4ccb791633f67a9e2a8b94b73d4d74ca3ecd7b760f731730ba0771fce394041300166a0e6a271eeecfe52aad52b554560bc51844b4321256c1458c5fd4b31013810c5dc24198160ae71d689fb78325efd2c306144a60c2917e7594ec4d5f68534813e0cc2ab7f494d88adae8cc358dfa2c1dc17f329d8c835a4e181cae177154ddd37fa6b6153eff4cc40e13112ae107905bd6effd36e47eed05727b8769cf9aaaa1a6cc541e59bc9c3f5ec926f1c38a77f3b2d11adb6f169f52508359d4d7875734e2b78561d94f4f7359656e18aa157b7bf1ed0bd980dcf9ab2ecea5d72dbd3bd3bff2da5e4249ef069a62671e091af36af07e50dfde24748669b7ebef36bee1249af8c382091ef34f7e5a9250651d78375b1f7a5cbb9fa86a9dc9ac8b86deef9e0053c16bb240e4d491b8807e529f82677e4af8becda10dce379e684a92cfbfb68c7e6479b2524c3c50f6a1f63603aa81f67b5c9468db0d27c3809f9afef3b9f49be31d4aadc581cd58bf8736940c490d404b067c31a02f1dce2ac2fb0fdd96debb2c6c4ce90d3e2e055da0ee00fbf66fd81486b4ac82fc06c3f762f959ffbedb76b4c1ff18bbc9c4e36b7f5ae1652b304a726dfc2e15fc892161000471dcaa813b06cedfa133cd386f63732bd9da6fd503ef2ca15abb98fb73021aaadbf6d25684ace37d1717b0ef25d03cbf3ece2cf490e53352a6f669f44565236aecd0cec1f0219055eeda0d5f933c3cb3a4a2fae57def227e9748e809f4e4bab348ed1869e9ef55d89ced4b0033f1e377901b524e27877a83a0457b832e0b311264be27a4fe60c493a5e3e28784abd1998579bb852a289c00565ecd164d633716b4c675099a403d059e307c4c11c48787e950af30f25d07832e073539cf772d2edd98991e39e85facea6560d447d665d1502295b81c9f6d7a9bd15c4440a193c933a6fc4979d38d081ef548a6a35bfcf44e8217a1155a0e65344f9aec3d8cfb745b643711c69c4595a0628c39b0bfcd6f38d19d8259378ac16bd489881293d4ceb962a20dacad29e4232fc4040c9c57a0a3a3afb27492631db6f290be6c40a026e234e5a39880aa51fbcf58abf35c1c47237ac6d7d40527d8ba3c58e5a73dfe2b4a2d7a6a16a1378dbec3d28821899bb11cc96b69140a3990ca143a945fc64f2a65eed8d7a3a961f5c0afca2ef39f568d041205cc911c18c48b1cce40f28403c2c9eb6962cbe3aa08c5e598170627bd197dd74e98f7a555e251b88c820bf1cc82fa95d491fbfa80551307d4b6c59d6004feab7e54a4407178d269edce0899f9c95c8082ebb04875695c7908b4be7acfddeae14d661377ec0671b37a8b84e72c24843e0ce695780993f6cb9c674d4a6ce2e72b1463e1b63619b427ab50ed53c422e620cb08f7e3bd83767208651e86b7d267db6bc48b85d0010e3908a4be7a622316db45c97425f1a6500ee5249eba412ad194cc1e7d615a7b046a0b0d84c586c38ab025be76b9468b17f4bb3c4a4f893ad8c9bdc833c325d62b3d3ad1bccf9282a651c34e9c184c9032b5e302adc100a0d1fca9584b35d2b0e71510b7e2e315cf1e32a6f030cd545434d5051e45b937eb51a017453033e179cc94151683caa12746d55959f132fe4592d489e58e92af88cc56a0256f47c2cf503cb0841bc09519b4bc42d249f3baf2d136fa0ef4e7d7f7c672ac6f64dc06fff6719b77077f9224ec4a1bffa78a0ada747edde1151331b6d4a6c489bd8b1eaa33c50e65f35208b752111d3896607e6b010e9e027bcf4d445b45611ae5bf8a4cacc9b1e75be2697f8ab3ef2383118632eb88b2f97556ab3edee10a8959f3ca55a5d20e51ad9b44bbfa750a371a07a2b0b85fd52cbe53bbf4d40c395ba6c26f63d3807c6b65072b1b3606f6f9c7cd6462dc064d129330f014df19761838a5e9e59a57946b1e907d0ba1dc1dd1f8dd3bd7eb0eb351b098f7fd64bb3935ac08ba598eb9eef746d1cb0bd257175639b1a97e49fb705082d6243ce94"))
                    (acl2::hex-string-to-bytes! "77236761950d47b374cd31f3ac68d5347092a37b63e84c3e1639bc59"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1796669a13a3a820a69a2b7f4206d71406dba0d8ef397ede6d00823c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a211e5d1e24dd8b638dcc0879d6dc5a84cb3cea92028e9fdaf487539"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "aab8b76572a544eaec4cd64bdb5c84f3c6674e9cea12e596afe88384"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "a22cb5c43e39024cd1b46d349e84c9ba81fc42a3323c7f8ebc8c1884"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "6d99e24ae740fb14d971811c6958965790a4e45afa342823a2451cb4"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "69af5ff4c109182dae8ec7792deb6fd309fe01bd0b81bd8df8208a73"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "cb4309313212a09afd1befe509427ecb3a644a9e42c63d74a216eaf3"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "e747c2429ba8fb94875c77f8c0b2d0b5cc8ffe8d92ff3aad800e1e1c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "633ec87b5455c12af54a3edb3589b8fa32056ea686b9205dd980477c"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "eab3e88d00f2f164e92e95a8c9cf3391037969944fd5cc400e95dc66dbb2377c63f80c4376c5d3a85e0e3be86fa264f8dfb34f24d89293f6ccba5159091069fe8d69eeff1b4dd8c38efbf9216eac33c8c11ce7de2347b0e7197052e9a368edcd100d77091611687c22fdb799ed7506c850bd1a09597c8ddb9ec0c75cd1432ab195ed8da89099ec9cc21c5d9a158346c1359dceefa1ba919a03b198fbbd288872c40f1926b16ecaec1561d46a0d605c7c17c4ecd6248724ebef4cad0e8c9f6fd7447d04c9a0176c31c4d30251fe94aebe9f3b5d070a6eae970a98eedd693047aabe3f34da861c3836767ad09910a5a652750907ca7af49767fd9d44af7a6478b1d95b93a08a0698c2d45f26886a1dabff4036d4679ca34820706395cee7c26ab53edb0865d55cf39c8689ed683584248ca8cbab259902278d0af3cb80d9d3867d94c67a6f8ea6b18d33000b7f98607bc257cf734d99999d757454cc0e781d32d38425d308de6d395d83cd6a16bc3b1769f24a1627c3f1139f568e82c0eb2ad77db6fffa76ed37f5d420f7f074c99c604924457bfb4091efd456aad3dd5b82de44d15cd701e28f4cb9af8687e3e60cbe2992c68a1138019adc1fafa9afefabadf07e4552f3e6faa6d6db8316a93b1299d2e7d82bb0d2ca7d7f57a5eb7cd02fae4964c0539220026bf149f5850ff136590dc98046223673ff85a794818d259648162cd8c228b92e04d3a4e4bc5d300066a0e1c8c3ad8d8487b5b685522a1766e2217d9287fbd806d7449259a2db1ff92953d6b78a9aff867fd82d7f489ace29e63000f88a53a6134d0cbb5a21d1e7a808f050558431b3004eb6b38b4a277166bb0c5c11b47a4bdccff2c9f22086af956c3df9ab97b9d27a5c17f1b0411728a6c97d3ae5ebb7eeeac3bf61268843a049be0476c0ac7c837c11c38ef613b31d70784b427d1851b7c3923b2d003f50e24b640b729dacc0ef8fd5813365a943754c303d1be02491cff1a52ed5872ef59c23c81b93487dce3d3a101b8806eb0f5bc2591c6e16f5bf5efc636001c33d7af75a50fb81cb09ee96062a5734e0c15ef55b5bb0e22875238b6a339958bd55261a776675e52e3667227a8bfee0fb147a6b9e9ec1201d8a047fc57fa0827c5c422d02714f6777ee83081e9bd83360caa9e6148b17cffd1eac6294ae3e28083c165a84d952aeb01888717ac3acd616cd348e43887998d118615b60283bf26adbeac784054663e09cbf84176944702503638ab5feb123a0606c92b0545f87a923069c8a5ff32d1062706a7cf766aa96567d03f256851772de15300c9c7190cdfab1021e4e20b54751e34a31d3db2c70adcc2080f6be1d4f7aa35dfc673bb4df4a3521c93a97a38d535f4dbf676a9dbbef960ae94824dc401c6040091e839ce6fad820128b1e9eea4f2e9bad19dda6800624e8b5b27efd4cac9202ba2712efeaf2e32782aa7e4e672e965b8353fbeec6d9bee36a9e96277c1282ca98ebb056f41a9e609ffd7f5c0bd6e0b6ca7c8b1e9550ab2392733203d73678c12ad224790a5a50f1f4ede870df3053d309fb56ddf55335cfb4991b386500bb0ff84538ff379a420206e38331e92439dbc83a24d4efcaa20f5978196f2e5995db8aaab29180c2bdbc5f451648339bd8f58fa0bcaa8a9d3b0d0878f3503b274c4d696a5bd6efba0746867c6a5cc57b7a4ae795e808cfb548ee490173829354298f1ceefb778b6a68c903f14716a5ae650157816e96c4f811dda47b5fae3ec674c62b9d5a1b5cba2a6d5f0b741edd1ae14368cd17fb734fdd00a24e807b4567751d1fde3f6b6b9394592131bb1b75a93838a0dfabe0e4a2f67fcae29e92af1d74ba7e1b740ebfdbedb6d1cde2b67342b0078a20f2ec1edaa283a59f819a84d2b8c5d3bbbfad607c7eabf746b296e1cc93d9abf8b4c829dd2a34b9eef03ccc6d8a67a326e9f0575c8f611c04f363c63127d58dd0b81b0d95d6b99135b42090d632c778c3ba7eab656700c76f6dc72b58b524b70ec78ed2e9ed3bb2dd95edd8f5f80fd9fcf900647d50af81efe768172984edb11ca9c981c0b9e1a64f5b4caab223644f38701d8b8e9096d1ec2bee928cde5904602673220480d4f4f2cee410f5fdf061e0e57e277f70de3fceff02a569099037440534bb9c8b3b242db65e11a6ccdb975a21cb403838af53fd2c4c56a84eb9d22e10b65670223fbe6ada61b5add94d790fecabdfa1303096e17a73af31f5fc5d880d9efa688c084b10039605f26c6d90670dc60aa2a08a8fe2b020b8ec468261b9c597b06c13bd5387f9a4085dcd6363cb4ec5cc23cfe4c76998fa78c28973662900477f64fd865c368435c7d743587aa175502a0c0667c4dcf8aaf79905f7f24086566f38938cf0876cee484617745e524b0e2e458d3b10a599d73d5b03348f661e6d88d01df796e17879959492de964798a8dfb6ac87332a432a53eecb55c4039234d314e3efd7ac727eb6a7f9bfaae724c54548ab73b3d5bd4a403ca580197fa85257ed2c9b395120a4b169973e2399b60e69e1d1291a0dd1e10c2bd9de174229701c41422346364a9490c0b38488c7182b526a67ceef394594c0a5ae9645b4dc352bfa6bb086c4baf65b9694b861d810c308ccb58dbbc3e20d62916dd02c3e8b4a8bfef06a8063377118d283033734677e94ed038c051270f4fdd65fb65246fc5644371978bc7c88add15cc88d9043e506fdb799f3558eb7f0965ec96d4136f8025f8b2d02c27576538d1cab7df76c51004b11beeb339f7eaa3710f620817447f6e3e476321bb8c90b55fa74fbf11a741787e2bc9c28ad8fcae7d42df5986ab1d9d8293a2e4eab497c30de6a3108b217f1b814d92dbb20b0f7af92b514107efe534742c21ee3b8bd4f26b0d645c39d0036e68e5ec67560aad99532d7a6711af50e3716562b6246ef2b3f58005faa43f88f19389391ddd54519a49ddbd341a560afb08f3d7d54bca5b984be30a8460cc6a45afabfb37be8485fe1b909c20a8705e1ba94c13b4184a8a1fccaae471e61341cb0bc2b57d7ff7d52566fa89c5327ac4a5e0e288daf05c08aaf65c165bba8089f246620a37a84aa9b0aa80288a5ef1f92a3e3bb9c28a7b41ca675b32aa34b0bd7bde0e4bfebdd98e2054a5daf8746256292fcc29033d24e00e8e3cabd60b005698312bfd4b2740fce503e0239819485623141859159ee54de7d984f477a41be007ee5fb0a58afd707f26d394d1b4208398a9a4bc520638f5d6e7b89ef7764fd42b73fdf8864f8b736f3be4a13deb6768fa5f05e57139ac726c273c2dee1838cd7350ab2775eb678035c187cb45159970e8829046db7f9c4e1816a403b3f053168002cbdfece50e21534651858fd93325c3d2f863096e94d35b4474d2f1a52b59d218f48463d30bd15838c662668cd1121cce82cc737d32cca62f22fe2f9d2df1d9f199a7733bdd1aa5175ae471c6dfa063b3e3d5a221d274993522e7b64ed799182b4fdb8365fe67da70838e3a1ddb6b294180313e031075a4f519423e30b49a0d347b731d07ec3358c2d4b217d830dcfdf5dc418f87593a49fd6a701ff9419f399b9110b8882317b09cc31293ac9c60222f7c991f25395d08acab13e3ff0af31dcf4536ee6d4f8f725cd30245f16abf567cd5aa75732d3851913f8012c87d303a4dfc58173dae050e818913a75d18250b5c7be81ad79176c539ac5734cc14f11fa62af6db6fdb12f5a959c92036414ba15cd3f497f92272d40475fbb2a07ea02e565585938a5323b842d9855913629dc8dc7232452e4cb6fe80310034bbf6b7f6ed0096e7f3b2c1d6dc6cfaaa8af895e289db33cb0d7cdda54b41e7039539b2f9c8b0ad41525677c1d4987bacf37fbe7eb3837b920d3f4178b43d22c7b2942bb777fa3b282cdab019d9f09ac8e36481fe07ef7c32b4007ce9599745f99ab28ca07921c304c99c054f4ae9da96d1438b00e583a805786cfa67b9c4c301b3f140555adbc78fdf36d1c2a608ad9f539933f7dae4a1b2f0d14484436a1c88a4fcf1368b9ccaf1c427b01489c1a63ad0e217736696dba383e9d5cf84ebd4f6e42229e489c940032eeb3ad554993701df9aafd406fec86fc7f3489a0cce6ec4d80d24bfe405c91c65fdac559513e27ce63f57ca3aa0fa602399f77429765c7b171aac4918b6c8a60fcbbf45d9b63b147b5cb9a68011d42d2d5a7cfe9998209217d71b109670b14892f1de0670f4a351055f4ff965b6084f5b914399ab984da260ceeb96fbb8357c760c73182953adcb2ad385435a21e92f24fae5c5fdfb5614786ea85ca6f1a7aa5f58290b3157dceb4235f414ebda09451fe3de13ff35bc70c532dbd226a19a5a85716a5047759329e21306de7942eb39b2d886f43debf33fddb7f087f96c8025aee4b177ca31730f90e74810b0f0f16574658af718629cea386a80eeada6eb64222e98e2382fef24c52f324bf168de70db07549d3d4e0c085bd6f786a1e54ece85cb7b402fa589d1f5575ff0d91ee4b45d3a7e471b02e302be69eee084ac3cad5027e2e20f46b5cfb7ec2bafc687b6c8817b628510359461e6bfb17cb470178907e06a7865981556637a1fc94837c725de469bf7db3aa2ce5b24f9cbbfb561fa13617c53a0fe0b0eae8dc6ddeca39854da2894efdaa690708256124e04f0ce7919479304600d0960ff739b6e126f9cbdfb9de80d7cede02102afb58432b6af7e7fef391b1129b2f5ccd817fa67d066f9b9130fa15095142842be3e0eb78b7782bde4f90842f88a259d27b6f33b9a69709924767bf4209f2ec8fe263e01c3c91e69b53d4c63ede188e6ef12f92c0226cb225eb6ddb614f38d1bec08a2b01416b9f9dc8067f27e6b8840aa967cb7cc2151db0e3b4734d6b3cb92045fcd30d0d2dde4c75cdb8d2e8abc20c6dbee4d3cdbad29df8e58ef0a5272f1d3961e02aea002b1024d278272b18d0de03dce7af8d855a52f2a835314421b81d63635b3f1e0c0f519bbdc70ef92837e6336adbf7489c3c3999a46ff89251de888ca84b34f2781177a335ad6f98027e8ec710c993472700b8b482442dd5e028c67cd8b8f1ae6dbaea04e2333251d02d3d8575b86ad24f2f390de9bcb956f61e3ffb9caa5d9a129012866d603f6cb7cd5a3742b066303b50236a82e0825db9495bd19c2a62fe901d58f17e3d7e31e1e54fa31eba6eab15ebde94f5b77132543083c57f3fe7210ee9220cbcd0d7e876fd8392b2dc6b890288d253e9496e1ced5c1da1edf2d54cb57ad9a0d02333d89d9d5ecba62043132cb0676e079f6264ad53932b904a12c7ad5ae675613e784a7f4e4818a18edf91bbea05aff6c03353bb9f4b2c0d4f87da1d11af53182790864497a93f4b2d274bd7bf14dc5eb4cb2f40c7af6b0f091ccd2d259e029de861d41301e9094d050388a6b6fcbee9841040635425b23c81bae707201a499789d57c1413703e4f1e5c925357251d3d37ecd04dcd17c73cbe4771d08a8af2af10d3c57155a7a6f88118af042af7c102d5462525baec73b8c8acb3b3e958c38c8742f94f4ff408f1352128e2b8db71e167058814a40c7e58a8d2f5d6c7eb85d7f50c5ed98ac02ee1425d45148837f781f413414212dfb0c365a07f61ababff3763183c348d74de71b1b64dc22274198c845798d68ccdd1a207fd987d1e00fb55d84edd8be076e14989138a160ffba0a720b9d5f89a3ea89c97d38a52a7523fd85c9688040c30ff10d25b4236d633537cce213fd2c59afe5a1b099b07ce099569f7d884c79808ff6f9478be4aff83fcb398771f8994499b839e8714c41b8dc3b0b2fc1afd0c68ab0dd851467b135561f486a3bd4890b347c025353db31747915494ea92cef1b7c993a5ce21f099104ce69f8e61353f40cab9bafb5df9499dc9fea68b3461cd5d22a804d799d3191b8331eb1e53782b59eb1564c1bdc139f83155efc547df2bc27ef5fc54f4491be0ce7e723cda3adc64737d2c6fa63170c193027781156f904ea521952d7b7c28c6f7753dd140a3d9cb2b9818ed5e71755eb5ac46a8e43a86695ed5d74b7694c735c4bd53c3cbdaaaf4d80f33232daad0395ffc6520fd68947b72b9cb942db88195ffffbcfcc408d749458982f95d040ab85b99dbd3a82043ecaa7478f1e711dd4f6ee281627354e83039af0f5835a6aa45702db0a72d8a1924ccafe74aeec763c9da0fa9df1fe766137b29e126fcb93847655dec86d3cfa08ba0111e8888019b6c0a2d3c5a11e8d959d49e004dfb643d2603d39857e3176089f0432791fee70ea0c8e543ba9280c76a0cb675b37ad040491237bc02a589e08782ec0a4f98a343cedadfc4e9e8d7baaeeb1a4742c024c91a5a305e6e614ea8eb1fb5cda99d5ac25c09c2cace5b14398d39727ee017896000a26788855f8d401f8f467a86573714b41d365a513560cd6fbd652377c5278e63b484412ee66f31899a4618d00c5fc03cdee57e5abce2fcbc409c3cee1325daab562e9251401c0193085eb06e05fd4e5e14743dd13ac351dc8abfd42d49accd41510d81d7b3bb3ba4a03549488a374781868ea09ef52a51376375f01cafe3ed8b6cca19a6887564232b74eb46e235320cfdd9adaa3640863593669aa38df3c5f2710d5fa70771067a72c29e505720976941fd6edb442c773fe13bd9aa2a03"))
                    (acl2::hex-string-to-bytes! "2ffa408e12424846b83274f71c3c682e1dcd13867058b01801a0df66"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "c429cc1d6ee0cbfa8da6540d8950baecf142a1401e85be76131adeb8a88e2a1ce8cfa91bb73e0644b80594316dab8fbba0e6a56a02b0b42f139cd656416b9bb654da952b09bf1a46fd544542cdd33e3a7c43bb9a1591c7d491e2c354fa28aab5d1f39935ae8b8e66263b6f27f1ef4fd34c02eeb89d517ecb5694ad991beedf8127c2bb21ecc9ea0ef9611bc821284beddaca43ace317627d2a599c0fece4c3821eee05d70ca1b7a5406f510da5a6b9efec44c45e2f0cfa52d4d9999db533dd217d207f3f553c7b93c10c17dec49021434b1778b7c957ceb31ebdab2bbd02b5c40eff304211c2c13f45e17723c80e0b50bfec679df8226606ceaddba24d7ad4d5973420c6865b0e13939678c4f74a007e106b9af905924bb4ad5030e8adc3c1bae16657b4fe2dec7675d222a99a698735053b18b438d10426834fb2e6250404ddb43edef5bf63e2d0b13d7ebbf89629dd6705e9b4d90719c318516984fd6526a4d864bc6f1123798a108093e05bf76ddefe7016e3305b12f233ff5eab7c0fae79147ff90c3e95370a5a436a0109b4dba19225a62885f6781d9e019841eb2a24c331379eb38de3a7b9572d09074bf6fb8cebcba16a1f00d3ab1ffe1f2f92dd11e9f836b14c45324effcee77ad869e55608d00b109347a0a9047f9ffc5eb951fc552e213dbeafadb6458919c55eaba8737e0b8dfa8a45fe8490858b36b5a8ae6d6e9ea347f3137966f14e6af07f5968da5462f867376143ebb6fb6ad51ba61bec9c0f62d9d2b9c1e66de6820525ba4c14f49ef55f82e39767e1a7e6e27afdc002474806fa43e8abbf1f190a0b37177da8c4872beb5312217f7653a563b9f3c636d71238b32ee95f7871aeaa0b9db8221e2ee4431292370abbdec54f63c0d15511b435a00b7b050699ca6f090c3fe564ff987fe4dd279a79a381c68de777df941a4779e50a1381c8aa9122d01605e66bd26c1eb752fa8397c0c71d07f9c664acd9f6a6afb8dca8cc343090a631d7025e1f118a637cf43a8c99199464a095aac5d62a83a81ebfc13bbf0e8134669d83656fef8b9160fb067fb6ddf14ce5ccf7059a4cd97643a4133cd193bd22bc3fafaa486ee8e908f83198a07dfd69ce539497acc270c12f74e7f8f4270b173ca5cf5235f66b127bd40101de5f80606612c9a194f61032eb158597af3234bbedb72a88bbefdab5c175bfdff8ed42dcb2f1dd4cdb4e81707e7135408a1c01b7d75e0a7264c1fe79e3c5b25d778d0a83bcd91abf08348a24f06f3f1b0e4d37b30280b538b7d49f07e8ab24a0e2a9a1fd8bb0c623a30ff5f64af6ea51c22ea156770674033c918506d80e5bd3c1a8d0f327602a9f3447888d93f0800a2f2506493dd98ffb7c5e771fbbd8a01c9df25d0d99cf8e5c44440c92d344476ddb9b2d345cdf4e47fea43efa9af03ba3fcef10f2e8e739815a6d48ac12da88a350f9dfbddd02f9104acca23e7c3e74916907b34b1c8e7537ff2c0684599a8ad316a5e48fc729d77de93c6b946aaa930e69a20cbf3eae88a6bae80cc2af6c9726b3090450223c31b7625e61a36b1cdf418628e5f7542ebbf5130a230d1d38f5c4b18b07a9f6224fe53d994c14877ca1534701049a69f6564d92f8130daae6edd49895b4414cc579300354f38ec6b2226c3495315182eea1d5c01ba46b5e5ec77fee8ed4c12b0cf7c03bf91fba31a6a7b2d64c36c4a9fee4b624db9ecf4188cbec3dde2fd09cdf10bc6def983fcf417b50b48fda9ae50e661747a1c8e05c4c41440cec8d17a6260e4c229e26c7a969e56c57c4e47c3f067021afd3eb47b69370aee422b63d51c9379ae06e91f7c5072c2f45551d11bdce4afbe6dbcfac5743a1534ce771eba5b1d4cb32c7a0792d0499c008a84d026a4e3ccf741f3cb548461c5acca388c9abc42c0a9d111df5e946744728273cf64b6f8ec2df09fa0d908229ececbfc17e6aeb897dabb4ce12ffb705ee2b38b1966a1787feb1ffd05a37e1103135a1e5b3ea0f314ce3187b79793e93db9b45758ace2f70ab716b8845464a18502eb2bc0f34166da0cda5eed25ec26836ffb551797fa091f0a8b4e875c4a48a72605a0d3b4fea85a1f2c4eec4b72dca28e0f00a0b865032f6dd42b993b9d9aebe5102a2de4371c176d9db42f78b6602d9c55e6be9ea299be5987dc2462ed0cc5d8b1041a1199569480b54b314e255577036673bbbb31214e462cdcd3729712c7696eeb32cd83cc43e0e9509c3c5e21315bb734128e9f2e41bc19aa00db5a41632f802bf2688501b816c080cab3af57cf2ddea46d995b354b33f4c0571d751d1b93bb62f3832b8b638ac1383754c76dc2c23341276bf3cacd4644bc10b9960a632e671ca09dd5cdd8c572038c8004d79ee16b96da66d7534a5b7ff9f8202b9ddc5829a57429124d8a5f496f14d60629c5e3ce11f11a946334fc0d1667f1a45be9c969b9f8a56f8eeaa0bdd77ac739d8ca42015ef607a97f3f90727dda9eaea90dcaadb6a8a35c069c3255d928c41e793e947bba73be950bacc156acea979919db047c27fe935648f1ac97d0995d326f7553b3d0d49fc59d1f07103ff26d0232b3ef4b8a8505fcca9d47fa8a6703044010f8301ea33bd9a808ca35838c9f58683ae3925ab67b9e1fe1ccf432fc56027f940a77105a034fc0ec934d3433af2a4ed299e7151d54e89d81842b76d3b6cb9270f2ff328396865c5ceabdbe720cf63b0d7ac04624323ad7df05f6c287b5be0d2a5a4f119a661beebc490d7eb2392ad16dfea47a658423636cdaeca282d358644298ed2a18e27f0c9d26c3118ec45696569e0a0e5bdb20a4bd7b83a34f80392c3c473fe889ac3e2a721c5503623ef7fe69f1778e6197b13043504221684ccec989edddd96713c7bcdd8269c0d7099d6e1a716387215a637a9101beb947a987c0bde53d793bb9aa58220147d64794a589ea9654651593960818f83fe77826f2a63ad5e544622e5470f21195b04e90aec64ce8d0c4b8c3f75d5dbc6c186435adeea3285bc784c2648bc02bc9eb666e17a7d3d15f02756bd5eed1b6b9e8b666080e6965077f1e16592d1ec86ae4098d8eb2f1f855caf61bd031178fbfc4c928af8a00047ed798aa80c9606fdf46a29f625c39c66b7a952464c891afd439370df2e6099f386d8eff99e5379abd6febe441181db3f69ded461219c28f9da26a8670a6e88dbabd5b75f304f6c18bdc202c897b00598da168d156a94ca782bcde12a7511f687c23a8ae3aad308ca12809a8ca09941f97219256a0fb72a019d52c33155920b8035cd128bf427744b3bfe31981285b8eb99ae4568f84e6d1c987cf8839439ee4d6a57f461f793d923a2056df2d7ba92f5b79f95180da0be5184122528184ec876cb9fdb6d995f5fd8ad5cf6a4d581f6a034b2bc8137f74329caddf529cb19c084be6f6db6c3d22df4c6cd75b6f6210c7378b063ab5c99242bd0a50d8d93e2fffead8d52445e356cff541e922e20e4fdf6eb60319f65cf8ae0ff31185c5af0ad6cffc2331f50d0437fbf18f4163d10f1b1f58ccf643be19dcebf4bbc9e078cf1561cd6b85083ebdbfe6e83447534ee0f2dfffae402c417847f79b9a15c82fef3456c9188deddce5c03cb272954edc680f16f731982eab9687b9344709eb38edcc157bb4cd148c1ed173dcb4cba50e06c85afbf16eb940e704877b4d16a8284d450451bab31ab0bb6c12c3954af4583fa7225a68245aa5fc5708bd16bcad727027c9e7a4e07edf5e0890037ea3cb4f8523f3bf0059913f48e7b3963e7c59ac08fcccaa0dc5af8c793f7228eb66785889d74586448eeb915f4efe831ec63b397a2c83155fe33b3c9f694f8065f3aa078039ae3ccf34c830a1f56ff0423ad0af464f93b615c99064232e7eb668ebded13ab3846eeaebb591a3637e8a434aa871ffde6518b9d94d97c6e786478eaf755a0bed6e8921f44238f611b0ad9c663135f9418d99c6a2c14bdd3d0c9021cf0be6663fc3182f1eec9afca89cd6b81de94fe51153d5d6a6aa04e897683cdcbb012aec3338985cac234c3960999210f25f4425d67e39346556377b7f2fba1a04037c00502586253c64a12f023d6c415401974e0953509a5f6467f86cd984543a717e453f206e23304f4e6fe60ec1c3e35eddb01289986305331b01d8ec1835bed9862d28afbcd12841b85e84744f12f0e56c1e9a779d18f760d50a5bde021901ba481ac52a0eb9a4da538d761440b93d33e6cc09a3cb36967c0a845e35ce88c37228b09d25b0371667c47f3345532108a8c25ad8a4af4662c340026879da0f44cf9c6109395df3fda86fa634ae02d4e3d2834d8fe0b5c2fcf4fa06f310b11735f137bea1cb745adaead99502b4bc5f6f3f9e16a7ee7b26c58fbcf49690e02af7a57f1bef527fe62ecd76be15dffc2b0b5e9748b9246c0c4cd1fd6dcb80e3d50a4ad72b5a74fd2079ceeb93f6a65ec238adf3462c3904ce038a3fd4fa57388ffe6fd44f7b9968844c4d9cb91dae40af87ac46b18982d1f68a06e8ccab9a5ac64472fb0abb363eac06e80e90033e3e6d78d7e5fd49a8501030e6acb777a0c7126cf2cfefa1e20be24a1cb5bb4ffcc4666ccf658bf4f00ba59897e027c72109eed86da760b8034cd1766a268d4c506f3503e14a0014107111319927801291afd63752260076407df42c4ed5a880ef9b53bebc17ec1f800cfd9c68f762f2690542ea3ec03154f01bc4dc39477fa610d53d0dce33927fb55bc52995b442df1dc6cf61821a85ecb8a2bdf6b7f084f46aa97b618631a5dd26ca02799df2a2412640d64dda797095f0af5a0c59754b245384795a72a698805af4300cf03703c1192fe1ff968183f94e9e23ad8c2a549e71f2c17921f6ed31c846d8341b05d9c77cd09d0744b7328f08fba52ace9413cb3c70ee2e0297064a055e55bf9cc3a6dcc7cf41f5737efebfd6445934af571b43299b59e9932b9631989742eff404fd6960988b3a916d5b143d14ba2b9cc7dd9eafbe8f4c6218ff90d88e2a559f5e4005fdb8d3572cf6da1493c932a0158add26556718b6ede3b4e06baf76711d24113ae5acb4cba4affe4970ee6d12b7f49c82b13160d995004898b877a1d2fbeae360cee0aa18a6fd3a1b5b00c3f3e2b98e2c298a01f4f6134d22557356eb9a3729100ff11dce92631012c88cd1d4243e349b44fd66b99f812535a92ee03b968d64d12f5a9a4e057602c8200f11340af7224899e82997efde531b5a06e8336f52292ca389aa176eaa949427f3fc83e578d9cc7ef6a510759f697bbe97d9739329e87b130ac7cb3c75480f16c80e547237cd9f8a9573a3bf28dabf6d6e2e7c90791554e46a6dd9fd6a9fb19317c1160fb8602a33412fc22b1f195edcd671aee5c452329e0526b4300efdf4135fbf593c7f53f51f86040c47e6c7e2837d2d8b019001acb6876cfa62b7057f2e0c539e76f0fba895d0cae2a42448e307deccb18260b62bb44dbb1d0407288b39aa18465db1d98d3cbdced5be3c2ec8c057d992d6bae2281de49b7ca283e9d0ccb597e13f36604b7aa9aa65f4b2ec1c4394deaf8d03083a69c888c9bc91cbc85dbb3d0fba03424be620c803f60d034618023677f3edbacb1e85fd64b0d530ea733782225c8cd206f5bfb24020cef6eea41c9b7cc9bf5c52a29973ee3db2dbaf939701b97547a2153fd2e39b036d596e7e60583e7d1c7f89ac6f86b6a1af0e8045662645b2d834ff0b16f6852c6c8d5e0d6a073d6c6996b88294a06e8b4e5c24ef217a5793930d1d1dd8459eea311535e79622ea364b3467cb955b66690546d61765885f762c9d07beb36b38fb7f837e837125d56915cb2fedee124ea968514c548ebacdf8f957378414d1e3ae9c4aa07787aa7f537ddd79700b5f77faebfdd00d298c339a8dde34e499144f921bfd342a460b58a3d7edcce879ad82ac04cf09c7b29720b94e9478869cacad3472fdb21238d2e1c8b87e92cdc586ff5b4727621aa64605f3d01c8ab0bbf5bd1445de92c5dc9bbb40940283865b763e7727a9715c913363c8a6d8e4196399df9fe9126ecbbdb04599ff1bf7a8b7f5d08e3ac7a023ffc9bab6374c0d2720038555e13b9e8a38db01335e1d67ab6770461ed4f486f74e838b2b06086e406dcd5ed4c2acba87ea568eab35996c14f6801f979ad28e5d09eb55c59770901bc951cc7b11f84b5b5ecd3e701a126d6bc1afbd9700c43ebbbe8b9ae470271b0fbf2b4ed733a0b96ae2a9a110701af4167559b07b8b5c1099d8cecb26048e67a2cbc46677cc389d3d3276471aec1598bb6e1898d3bf0c9c7de53f76f8d938fe5b5bb793bb20225f58baada0ddc284b3b13856c73df0483aee60582edb729947bffb7f673ff3f08833d7da4ef7a7faed9e50aecbf8e3b18451d6a3435c4531ec4727864174ac159055366691f710ea9ed9439b27b2fdbb014ec49b64e9e59554776c56c2d7bd14104342ed1a3a71d650088d866aeaa2d72a4b96caf402b83608c7468b6bf79b7e807489d868506f40a9928afb061e779877ced99c8145519671bd227e9d42ebb7596f77a2dcd75dbbf41de88369bcd071725f199a9d10262947dfacbad7581251139a98f7181db19ac7f1229a364461e49252c34f97f1ca70d39521caca95a3ce379c6a2f7ab22f7fadb2a5b9dc43a074506053161278a30fad744e18b78a14914ddb67c6a5231bdd1dd8fe6ccc48f0b344fe65a88cf344584df37e4f5ea3e94cbeee1e5707ff0b3580ea992da6b2e703e8936751782022ed3da93eb90081a653132b0a4b1394720b8f5e0276cc2ff7e8a3b5c4bb545dc320e50bf"))
                    (acl2::hex-string-to-bytes! "a8d270a4267161167dc6e0b1427a8b8476645be263d29c5b8af924db"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "8b8a54913b5a4c09432ef2faa98ea704e6bf068af26b35cb22c4586b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "662eca1e9e98d532113093abc2cd6300c6af1fbd159d29210727d4e2"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "7f28550790116649a20517a4cd930792e2c349209d7d2d366481669f"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "078f1c8cb726ccee15853d9f94d8f6681505175a3b8acb071d5e6c7b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "825362c8fbb680c9485933a84e343eeb488634cbc9e55b020c93b369"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "fe345afe05c43be99842c395b02e515e02c96cdda43210344f70ca6b"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "3bf13f77155a5d4a56b055b84862e45bb07413e801db0ab2d02b296f886f553f5f5890ea889bcd69aa9402a9922f7e8b25087d2f98d1fa72e3f5c4b5727ad4e11e2cdea75df8260db5969526776b0c20a9adf88544c827baa000dacc2c581b6b62b35037fa3a113b5efd37b75fdd352326a9ac706d873991ef96923a1cd1d1e812370f21c92935fd8d130fe5b24ea29b8345c07a7bd7d900b8fd1489acad9e5d16fe42ed308c3b5a2a1e4637d886fb90a8c95c2bd680af0de37e894043b32cd456501224992cca5b1e670336394de70b7d7daa0ae64a62c3586119e848ac2054b977a74d6bd28b064f366f683d6b0e12e9a8cf9f14fd9b27ffeeeb08256125e629ba28f00e442565a6d3b2574da719b0d794f672c6600143c06d28284197e1918e60be96ea4b41800b85f38bcfcab0ea2c87450c653c3d1e8425f7d10419fc5dceb290c667ae875d67955a8b2d9c7acad1cae4f45a59f6c0315f8189d6ae1f52d2f8c708ed379e06a04b10dfcde52057aea5fcb20c7fd532b163eddb1f1df279827a8ae4e4ac2f83a0ef3d0f196d4999ab5fb5cfe276817bea79644c472f1cd9c67f02e3b4d6e1da4dc908a377fc9e99b14236d24605a3a3c6e90e66b6ac467a8751568cc57d5d1591219a245159c75ea694a8439ff6e3b32cfa12b0d55c1765ab36af677f694b3c533e1b1a0e4d6bf0749e32ab540fe8b203f599b20a0638a11c7735fd4b6151557b69b20b16a0325801a8fd90cf195ba83522caf5be745036c05b0fd58f2e1a6af0ea8781d11ac69d37fa317d77dbe39da9c9ce005a6c082d812c6678b4984c829c09c303a782ccc7fcdbf5f7b41f299a4732cc5b44b7088d4630616c236790a6660e8e1a00a2efdf3a957188eeabd5a6f7a5ed8fe61df75ef6b1293b43638ddd36400f4c4440d5ecd86eceb78d488ecaa1695923a7657da97bf5e56c7b54ec2ff4e86f24db2526f6beaf340cf9befc0d2dc7379fee594375bc45be88350154a4884ed5272d198b06d49fca109bd5c8778307af8c25bff1caa1e8dc16d8336e45dbba6adfa2e4ab220288b94ad9559a108c71eb71b3602fb01228669eccb2e219d37456d3a34d305530b028431c0519e12e4c10d60004a330f74b85a97846572859c3c08b961dbfa446604081cdaeeef152e249437472f851389a60be46b7757b56f1d090e51c03192ce0298e5218c148e179bc417774d41cbe5d20659dcce519b044b069f612a1bbca780af2150b39efc15a915771d0e0976fdcec26e54e4f0a5105859e33c7449208700b337c4ffeeddccdf10eca4fc3cc345eed8a87065a571f7b7ab8030392a7cf0f56dbf1abb94a181904edeab925c3f62d4a18e9b9af0704f0b43ecd8d3301059f1072dbe9120f5f88f17c39d6b0839b8f2c597432f622e7aa5eb8fb2ed0a89466647ea2cfb0439860e143206e0a140ccdbbd39038d6b1aff29eaef4ae11a562ad7889ad263ceeabeafbcb0548db7e322e1e10a692f7bb46109ca920d59dff59bb741a66b55d187f6e689b512dccc9f4fc7e981c090248934c8231e76a1dbe60da81ba3446bccafe344f3b6de584fe397d02f15c659f617eee75ccb13738eb5e50c7ef7f9f000c0f83d934bf38ac25ebfbd3a5cc6e750cef3d9f04e602978d3d666d44b089d2e49aa52d0db1eeec44f1bc183cd6eb2d53ce822860ee94e1ace4649b3e2dccda5c88e0754eaaceeb653309fa3ce1a66e3e0306386831b5bf503b927b675f30bf7bbf7dd68012a1528f9929a12f5465e1d4770ef5a5182cd9822a373246ae7d2a4928c37a2a502fc8cb0b66349c99f12e1d10e58a9e09c3d649999deb63837ac2cfabfc89d4ce40f6207106dc0c1712fc21b7d21ddb6d405fae700acc4744f03014316e2a06f39c8496b352c006368ff3e9551cd7e5ad6c1bb897c7fd795ba81b20a0638649270a1935b99209125e32bc7cd00cdf432f0fb36ff61ca7812e672cfce221c524d2545d066b0edeb8f7c1ffe212ea5e7d5df1cedb7650f2e827f80bafddf00e205c40fa10b20d68816905632401fe7b1c9ed91505fdad0535ac225f417b1a89c32e97955b74d9cef41dc69149878f5fa759912ff4747909f027acd0bba2e66098ae20d9f09bb8399a79cc31598be631a6f8dfc6b0c4fd25a2ff5f08243c9bb82445a56bd60df3d1f9e6deea5df44c1aa2e28c7f42d38182d29c03ac843082bcf15becf615b4d1bd9bcdc6645bac0742227f6004b18b4d4708666067e466e3918659bf20b6769c2608944a485e9a5fd0ae3722fd05d3a89dccc53dde4bc606626ba5e491c03d9a33c6a5333e6ba69eecce729c114792dcaca72163bcc138325529a923f23bf0150d5f6648ed8cb80ac00b9894e97eb22f9b20e29ba74ea4b09247cc46eb307753db5b4153fec8b8ed71ec11d32db21aea93c070ab47fbf2c9424d751323a6ba9dc6409d1a18288e8d2afa452184ebb254bb0d335b46cf0ac28dc30e5443a5a03af698e167e4b2233311bb9d45963262b73c91947549d3d4e69a32fecff235137492ea77000a705d264eb97f090c8e12593dd4f9bffb0d397c9222ad4dd95ffb91c3bfc6ad316a25b00a7c54338e08aa84e75dcb0301871fcb9c198395a02a3edab09e363801a764176d8a858cc3a7127e62b473070649701ce1a1e871ee9fc597820f2f67242f00a49e6d78dc5473338fd5615794b05912b7894144a76bc47810609a19043f0ae39fc6ad97ebe874f30168ece171a7c1efe4bb82ff00c0a33f8c73c85cd5fd41f3500cd0498028b33b1a45de78689489edeb9fea814495440355e22060a303da1afe9960f565bc37f42491bf833f1fb7f6135e03323b91adaab1d62b8e35fce03ef394e505604b08046577156cbfef55f1a620557bac5de81ee9776f4c558e58497526e5e03a7d2f55261d6a9ab0ad6ac5ea0c66c261a1fd4544ec7603c9614c22cd77eaeebbfcb3cdaeab4575da00ff5c284bc1702004f2090c5ec81c5a1ec3b143ae7b97aa6d9c56474a2f6504911b9da704ebeeb22ed15187f4ebfdd06dc6de1ef42143d07bf3d73722afd2f808c670b1606da3dfc8a204ffd76910f3f55c91a870625af6ccbf37c75a4d247c0c08bd8c87d0f66e7a6b0cba2ecabc8b5f8db2274bc944e20808603f1a91c35945ea6c81a09c5129eb3650012294685f526129e4acdb695dd8d639e78c0418575c7366d8d4571bd6c40ab81bdb517e3877e1de27bc22322ee432aa06d52d893df9ac91d94a8bbb3eacf02c52808865de7614ba38a225051fd3239be620947fd5c39bd6cc2bbf9b294cee94ec3ac28519f603a35e876c4bf64db2f4cd93c263e5634a1ef89e7c225f4c0fd832308646b19fb482e293806d7cfce69d04563e9b3680c14fa6cc49b8a75358e16c8b5c106d1b31400cede8d17985c77fd392909190c307521edcaec5ca09d395e272cc4b160f82c324be2ce1b5935d80816f41abbad42310a8f733b1911787dddb0e5f5f7426adf5722dd504f7ccb1895814d8537e5a20708c4561e5c1257ca2a1869cb44f06bcbd3e636aa108d8ca839eaaddf0a2fe035911edbfc725dabbca7a030cb26f57b602cf887322b160b0549c58d9e9ef67d6b58018d6c084df4b3eafad35c29cc2a4ae6ec378eeb80c6ce69a4753a82d3da16b21ed2c39a5db34e787aba14a9dee8da8633c026de583051fe09ed394242a6a633c954b9851777fac4c26d8b43b8579d8f61c9807026e83e9b586e1159bd43b851937bb46505b3374444560663e2c71266f8bc324d574ef2b632706935ce31bb900fc02ac6a2e5829b931c3ca6e8d7b7a15b00a2c283e50cd30a8c9762d0013e7380caa316799f7b2f11189469250a2bc31b4f587c173aeb66532b327d7d329f70a8e3b3c0e4316f7350e3844741debd40bbe0c9d68bc11d049ec4b0c220367d1beaa3b3728d5c0e0cc505a2bda9a60aaebf7d8d7e7c1820226ff198ed539d55d9c8a34897df6fa5500fe76638d7208271987e09b052908344292a515ba1c7a84eb53ad9ad54fce3e17411554554beb31dc95ac52899dac4ee7ce780a97555e2cf5302fd2f88743835a40be66f9ce12b17e49c5653cc475395bdb19d3358b893e00474199ae47610c1124f90143a2849045ccb23ae6476bdbe6e010da5c6704a355c06b57be3731c184144e8df96f49b47a46fb3b4fcd961dd205e686d390d6413cc43574f1feba8bfa394672e4df6a0764de29cf3effffe76574223e64af0702fc7b28580f450550bdba6818cc98ee704dbdc35d7c85ad22e9524c6cd56ca0cddf0ccb6be0b318fa13c03de77942ddd67b0b514fd206a8b4a4caec77ef6ea7892b9fa6162bf1fd6f45734c48b286402337ef5c0c052af987df018bc77a5e2720295e42bfbaa6bc999afa1da1400c1e3ec9916bfce09aa782e122b0c43ee32579216f26d0537b5b5e5c7e385bec7b7f0f791a06872afeba8cf453e4d603a6ab7c5bfadaafbf18e4a8d12626df7c7b6b7e8fb2b515aafd3b8f5ebcae65904fc0111e28e7c6c6245e5e431b6e94b8b6eb735280010e29403172688796d9ae856d20b420d47b2f48d268f88498e16ad106d60bad71da6f6fa41d5ad9249a25f0b991630b2938c42b47ae398a2c305877c58f03608ee144319c777cdde110bd39ac4c74116f644b76c5174d55b3e84b4e2e867fc2de8c702e226a15b854a33d084e3a2f592d668dd0558d7e0cb5700bfe4ca4cafbeb41ba0f3dde6f9506bc6f04a841db0ceba852e2fdf812d8b832603024cf2b323d69ec7cf6b9d09a7dc119d6c14ac15132916153c73e52e04695128cc0f0e9905a6dcc1c54e481ef431c0426a13e2a51888984ccbb48864de601e8b927f04d3cccb323985f47cabde033f89b51035c3478f881f5e0fef64621deac6061b59f5e1ba5b8971fcaf22cda70bdddd883a021874621753e48e9f47d95242877292720cdaa55c29d564363bb3a1a953ea43d403fdc2cf5dd0fa97e848c19ba70332ca86c29c0c1e0d1038b0af8d86b8c507dba27206a1cc6397c0dbe2c52bd54baf81e808cc23bf0257bcb5014bf06c96ea6b7c77ba39300b0abc9b2dd99da2ba8d1ab8f3a23d9d638f34c14f5970ffde6015ea8569e01935a7dd9e808b4b9fee40077239bab4299f4b335de03282d30a0d6d85117240f88213c66e8136c01d456cca9c317221e413f02af37a63a096b836d4d4a28fd65e6ec2e973b68895ab6e67f69f8164f17d7032ae2a546944ab5790de41ceff1aab5a29d95686dd36eab7c60e14800fbd3612771f5b24bff99176affc54ed6c886fd2dcf87f78c12e2bb00b6363e3deefaea932011e61d439e983c8c19684349a3c7740d96702c0d61ee5aca935f35f035772f7e5dc824280040d8fa08f3a03da61a5f9eea5cb48d914d72e41e3551c92c12e3d502e0c97668172691b976546c3affd6ccaa438e5a1adf304bff47f74086c2976b27b672f47d260e9b5579c79e057dc1ddffab213ca5ad7d495f1d901feb90338af5d3896be874795e0384619baf9c081a101ed5fbdabd3f246640852c05829f7894be74a350094d23ec208a2ae16494cb0623125e601960253b373c3079a3c37d2bff9d645eee743237d4aa1543de4b380843402ee0c9c694078a1ff2f2f887e3e21f6a76ba2ad4d30d2929227c1116a83f04c73301d35bb9f1c2953c168bd4f0e2a545a00bc5574e96a66f1a3506de65d2f110356c4105db226605d48d6390ee531995abf9c8e4eaf9d3e4c415c389b5ea8f81fcc81bae44c1fed9830088a5235d04c1602e3a62f15f38f1eac55200e1af68834fbe673d3af71278111703f9d826ec33844f23bba9ef5efc52ecbb2c3048037ff8f62f9a06ec2b4a4679db7654d95c52e2078cd65bd2cc6fc53304f4ecdee8dbbd177aadf0c7d0b7eacf1803e19cd98641fb7c9b4e2f0ab00b42373f5b98a00167ca515e3fb18a27c4c0c387c597ed3768d6fbf2cc2fa129ecc05bf7c79fb257beb6bb953dd1e0029df5e2ec49f41fcd203e11c31bfa184030557903761aa6478984c03a30505ab03d158019b5d8f7b031dbc7f684b7f75f9eaa984bb132053792db6835615580996cc9a2f6516dfe49635724734f8fc7ed0067c0b7bf0f55919f64bc6e1809db26f2b9526b64dbbbfe7e573022c4a90e521c34f92fead821bc79c6c3cd9d93f2bd968c9760384ab7384986f4ec2b2e57f5bab74c9b78b6ee20e8aa853b371c4dc821eb7abd13e178608d8977b7cb452740a49bc0e3b2d04625b3b80b9e2fcc75ad5c7d016a3ec888dc120bc26ded17089aa1865eefe04fb033a4975ce0cdcaf1d60ceb364f01286f16a0b2dbf4d9cb816b78212da74a5ab5b281f080e7593f49a9fcdc2155a329b374ff26cc36d78a1b1cf21335f4371adb0a136a809d1b520690f3f9c9626ef2de60f6109d192f7c5c13d7047ebcf1b6ada322ae63e85b2caff2b8ea365fa8ac78414f18548728c8ceb9ef9e11e9be57b0e6766331b260f4cc36145670359a7e077fd0e5fb670b97f9760186f561e7256e39d632f3e273d3e6c6cfe580ca8c911582422099a0201064b3d767320eb3476e050bc3b060dcf18409b1406d08c60c29b16c70371caf5c97d637e7d1d66ce638b3debd7014e3752e67820f3fd68f18cc76187edcc200a21a49eaaed9a3582955a914c93be6d0886c4809f9fe1b31dd84380cf2907a5b0b7672b865525cc1394c51788e1abeea51c7140fa7b3c1f282dc6011735b2c07bf32bb68b444e9ed3da512d1b514847af9c6335cc166af1b305954efa946c04de5d68bc442b832255ee953e6e53220c8a7f20eb8ee1dcdf30e857d6e3d581f75a5ea4992b8b51eb0c49f7e80c77b7a1c84d10bebd6e203c445e74356d79a2fc728f75f9bbd4497d8964ffa6702f77abeb8ed93bc22bc39d4fc070f6e1e12d98a968534a7fa6c85e3fd812f633fae09f56ace0ff47ab4576d5c6e477fbbca0608171a307e1735ce1a20be853c26a90d0bc2fae55b85d045d58da3555155ca9c87d140bbe7bba75bc5288d64b95eef6f46e9fb7fb9ec1aeb54b62e9fac95188550ef7e3dab8ecb142c3c7734a4908818ef075943d6569a3234ff988f5823bea7ab975c1e923aa8e19685035dc3637def4d439a9b7a6c5700bb91688b1ca6c4ecdff263c6646446ab4d2ab947df80d5dbaac648ee6cd6593f339426fdc2c7303a36271a9ee9b6ae589754215abf8e98c64cb8eaa3753ba8d5e22f5b4c102d0c7bf65e0a0ce9d8e7d2bd2bc4ff2d36ce977c4009245d444dac118b8ee0cf80609668c1b9792b72f9ff06d6299cef20803682862cd8ec984faa07d04721c66a3273d475b14e782bedc9bc7ed89bcef3b68aa6b3db27e9e0b6b90e250e77bf918cfa82a57af198abca17254a5a629258767d30ef9f791ec38ac06cb26c4a809d7513b9779acc539b98d08b63365b03f6caf66761fe3651ead4a66aa0e802d337f6a6c3139b4571a360de34f0f4ece876b9de953282c64bfc9c5550170b077e07159be6a205c6b96a2928691a1d7f0b7f4b1b0089b392d4a2e8b4663e5bc149b4dcda6b4c5eb6f499fb967109387b32ba8996f08f1800b484c2abf33e6299fee7534b9a38307dead7de1096e4cd4826ce7a37af6c3979b2493122413fa8f5634781d7ba8f028363cddb7e1ed4644eaf6427e784479e6697993e019668b43e427d6e6415785d2f2723f9c8520086f49424968cb648d02d2b1eef303be10edd613022dbf11cc8ebfd8041d77a91baa49fb5d10f8e995638367d4e9878d33e388809a62597a3cb1faf8cb991deae89c6"))
                    (acl2::hex-string-to-bytes! "35eee77b8adf8f39931ce0264563f68b51b36614d6f44735e155f023"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "fc2503d7b8baebc96e4af56824ec469b3ad8829aba34c239a776ae91"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "ab103523d6f3d051bd1d48b3c9370269d9842a611a83bae98e5d4e87"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "1d5c7c600564a3cbcecda0efe08f5d375ee8135a7414b3bf9db6d53e"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "9e12e074960f9d3d59a9363bbec9625d7a6f1f5ac0343b043b57ed7d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "d12c093caa6eac327c6a36dc80ffa41b3905029ed6d9efb4147fbfb1"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "fffd444e93ab3d40a103e64d5dbfb6dc9cf7a23378de0c065ed8513d"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "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"))
                    (acl2::hex-string-to-bytes! "b3327b383d73bc7b18368f4180f63e26b5819525980645c6943c6eac"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "984e1a22313c6444064cca92e2ab11c75f1b4ac5aca1b2e48e7dd68aa55ffbfcf1d8bc73950ff573dab5e058763b7e320f4239d2fb53c7254ad051c1062ad5beb955c9c7307901707febd2ca455b7836314fb576c5d0bb0a5a624cb9653a206ef8ac87458ce34fe6fdd4e812d674c67bcf2907d9947f563ac81d0f994af7a3b3b7c53f1630b3a87b5d5a6a55b1ef31aaf0ba7722efca5f5e9ce8e18a3dc92836fd883861a453d4d7a6649fbe5f32816b9de94c7a5f18a01ddcaa0cb4c718759ed2ddbc4f71299ba3e0d9d07267a77e65dd9ed0086bd2df20924dd63e6f4c54943eac11081e9fc58713a3459c51b5ef41b8c149f59b5ee50ec5b88851becd8ac04add80b3331b192a48a94662a6c39ea6363b006877257a907ed369143b04e2c9fd5851793807603587d31beced3b513d60f23d8a888f654ec486c3b06e5723586005cc81b6ca624fd6090b63ae84d1ae3dcf4882550570ef9fb9dc4cdf2f141479cc39f435cce7213f335fef7206e6a0d5ba687966ed611c1754fe1117f57fa65296dff93b75b753c93960b7bf2957bb319ce69744b0017c353f6f279d0f2ab5f34907b9522d998c7c0353e42055eea8585b0a0376b718b8006377b5f9e7ddea62cff95b015c5774617a839d1af2710f52e11ce684696e7781a660e3b4e362eda90efd08d16ab7b47f84370b3768a99728146467f72dca62bef170db556d8065d0f05be848bf82f4df0ab656fe1f5ee9e1de2aaf566df620c12df1c264ecfefadd5c5a22f0c37fdb87f549a5f78058ad8526b5e52990b85a924029c369c8a555da3943df51df7812812b3820abf15e8a1dd44e32f9fed9b837146103ed683d1ece715e46f1793341d596cbf1a1db3a28b0cf3dbe4c1e21e1ba8fe0ad78368efcc2ebf3805afb7a0f891d4a3d61a7b2304601cab0ad5a577e229bf0e790995f98bd8f4ab414cb1324a7a7fc2f74bb8f4ae7513d31a1194536781fd7d9bff9799ac745ab0ba553c629361e2aec6181981d9ca7dd68b2c4b1b2c302dead8cbfa5a9e8f55dad97f95dc63f691c9fb05f97526cebf37e67c1649a0b2e4d887f51a96d2987a9c172279ea2c9701a6eeabf52906005c79a48792695598a6c9421184d9091e3d76fbc445366dc1b6c81960893807fdea1e5de54be4bb4fe82f9f97c5bb729f22fe1ffb842b9805259013ca220cad15cc987b0bbf6652472d9df90e6998cf89af83cc2ea3444befd2a1665f5c1155e6886a74714948bb1ceb76d6cbcd1b706c47cab8e44f0af9d0428e7986940092feb226d29c8616464fa65cb1767c9e05b590154ea2d40a1264f989d5d66644f4bcbe302e040259944df2b2219504aae003fd05f5e0deac260c6c55f3c54f48fbaf2128ef4e3a8d15963509af8de1bc9fc6031f5724db7bb5352f656be9bb9708546f638eb18b1b5ac6f1e5a3e7806da57a26b3eaf536f3407d972aa6aa37b9bb3b1d9294f490aed44ecdb8b7288d18919efc8619c88dad485448282c8d401b84c1a0189a7dd79202ac0d0541d75457d4b2e8e86c254bf44d47c6e066197156e5968b77f7c3cc7f664d545dec6a72a5ff54706d7a720e82cfc687e60bc527d65e085433355cebc19fc118250f58648d61d217879d5521b3206effab5626f0246ec58edcebafa1f2ef95eced3337a668bcda62c97d4542101f9fb2930d9546cb982c9fd78332bbae064a3738f06eb4e250cbd69572a8cc23fdd7354f6ab6b8365acc0ba5849c653d444c8b7abf941c64de34e79e16164c595ba3ec3325f1a0ff371256e978334b3b5a1b4e35091e497ab5842a25e0142f4aa9946be0a7ecbd70adae05f5c6adda22984513ea5583c89e6767f2df69090152781728da68fe5a453eddad5d08429e7d8f48612d3d5b263a742ee62d098b5f1b37baa772f1ad03e76b4f91a1b5c814c6cf93ba50190acef95c6778815a01950f78ed6c5cda8f36d1a37267564774b978a9b5841581293f3590903e8ec3124fa9ac2dfc0a7c0b8f9e77d4a2dd35ff091dce3c62c00e2d0389a1378cb7ed23eb6451fabade6ea86a9878e4b8b929bd9020ad86bae2088c98601edc077f30bcf4da0ae23d7b1aec6469f7213d7c6b36db2bc823011e08de8a4369ec13572643f8852762f853195de98da3188599a2f136899eafd9b6978945ce02e654c2366f62d047fb8fe8e263767ddd49458349835ee4917bb96e3d13474a7a692062f985d045dd2a48befea397b53631e1ebc763a0a75115048762132b5350d32a9be82dad70676dd747c674c9ce2e40df15b4f8a85c5e1a6fa48eef5e72f6b2c7ef7add66ca67349015f27d99132614b1a38ad8d26e08ea4f384c3e5d240c868cd03a0ba8ce6d2d27abdb24623ece1bfe07d17abb76f20317d50256a3d0e2dcede0ae3ecff6e9a5f26c18efd1ab86d518d4f164cfe19a037ded49eecf241a146f833ffd68de5d3d74e9045643e3d0ce968d831d8c587159d5f3ab5d3b7822db864a67bdefef486243a555fe55624c66fb840f0b7d260874b48a7f3875579d80c6b8cdf6e1942ba2d323662e72c14a3cbc6afdb51ed4dff28e145de7094cdaa2a530db4eda77877da7e5bd8d286b6ace6da4c7b0ee0f6831e713c00f73b26abfa645081c3703dc51365ed480d2db2e8a63b3f575772021e1442b4e182142ee465e87ece573c3aadc030143b02e9af60f433b1f7926a1ad5f39ad4077be58a5c5125be2413d31636c928f7381881551ab908cae42991e5b2a1c2c4fc4877e51ba86439095efd7a691692bfc7b7bf4e20db03b1a582af0cbc0702ad485392e52a50ed1608a0810e4c90b9937047693f7a446c8fc2078c36fb0da1b77102fd329033087ea6422879ea1de49cbc3048d32eda8b8bc41eef0c789ce90901bf82755537fd996daaba6448a65a0232db24b04687a7ce3231a1353a9a20cc80f1e7568246138ee441ff76701464ec5baec57ec9310a7f94a76b5a49f18f2774fff57c1e479486936fbe58694d4a82bd12f2e6c0c2fac773e81323ae877e43675374903121c6a8039dc1b13b3b54eee270758f3221c0ce555c613770463547e0542c39c4bef21dbb31ce2aefc5ca55522e14232736a4bbc56b1f3f9714fb5c686ae520dd055f0c4395ae6b8bd2f3e293b69847a8e9edb6487d13e22cdb1aaa86fd66ed87de579e242f291b1a4c18ba2f6ec8a5bb22e5e968fa06c93682d858c2757847c73120fbe56906e2a11a8108aec06ee56bc9d27aac4ac19cbdf17e8fa43be3d6a7e6034a0eaa0f6a2a2476f2961350afbdc6fd919cd50902ca105376b5b56f432ee648b428590a58bc523fa071ce38a9bb01ae57c103a9a363c7a1367c4ce33664173760bae0c057608c233bc65bec7e6c632df3a23d672893feaef681680956526218764ca6f90d1dba6beb1e2134d4d8d2c18ec8e90d395f9dc4df4d72a74cffaf322066137c22fbc460d70088da540837b350b971617dd18c291967d086bf74e7ce84918f133f02924728f4f8a4a5037b4b60ff3b2c9df0b325cb57ec6e58b7f034ca5482840c3b0c6b0582d95c6fd184d1e722b13540c4fa021da62d27bfd2b4ac1b38549830ad1b13973393c85c7b3c8d0ca839460f564a0638dbfa789bcc1a7e9fc46d175460a586068e8eb112088ac7f81b3d25b2ad897b447ebd2a95ae09f17d8036c91f5d930d791c33de398055c6d202249bd0d1a1a76818dd05547f6b4edef415763aabcef01863e8197aec87d3c3e6567011cb78d3cd30c2d1814727dde9a973e54c523d770ced16287d530736f34e3b6c1fa2f8469b4ed456c30f25e6c726b9dffe5bc7ea7d4226d4e9bdbd3789c23b579f5e084af9fa7b5cc3de4db19ff34d2cf8cd569092e87ee56632911eaecabef83c088f250fbd1faa28bc87869a999994e1b0d1b2746988239ebb6b3289ef2c189ca77720061e28099bb1bbace09f03df610d1f984b4dab56365f33bc43b8ed7f6743a78306a6fe41c4c6d16c988df5d422eb2a8a3e1a4e45d927d10403ae69f546e5079636db091a4a02b10eeaecd63cc952d9d0755adbb92d192d52f2570842e0937fa1bf47b15cd66e43daff420edf014a14b11994b97ada4030ee383e332ddf57f7ebc9725fce0f107017ff1a058570dcc3765251505089a1c01ffce85080e83957bd7be379399ef21e0f12f922f6be0fcf6847180803e246c62f66dfc901a640875ffd2269652a1c36085e69cff994ec1ecc2171495a8c362b649faa0479be746f1978b7951a5c94a14b0ef174589a7999ed07e47b1e3079d507cfd16bafdbddf833142b6720dc644520481a77d977c0281222fa06177518ae5f1ddc646da38408f4f804453e51940cce346a4145d3c9b4829a3a520d041a9db6b2545979be7088a54dcd993e632add9ffc6b3a63589a2386962dc7c884878952b6c1b19cd5a9618e3a0bcae6bb309fd9c5a5cafdac3d5f663e59ce0f9e675c7c50a9d49e903879f7e84840ff01ad3cc5ba9f5ad977ba4e6abb48cdacfac3fd7ca8962a479624ba287e0b6350ddaa9bab00e150c16217dada35822e653fa2549ae70f1b5f90750b6825dd9ba3f52ea4d3c2f3167a145eeafdd38d5703150d491e359d0fad25345301054550bef4413f84265e12cf3697651cbef3b24e260bc44fb9c66efafe687c3945cbea6a80eb540854a9ae8770c8357f1e3ca50d17a4ad779c15a00ec2196d590ef7d16f4135cba5755c516969fb416e1aee9a0a93e0ecdc050f0db11a4f0ea887526d95e1268ca4b1ff623fca377131da93f80b40c869c61d3e1dabae134770d96327a405c7804d2f1b4bcf3a85ad0f87b227d9c6630d7f0c2939c3256c00f72bb26563571e8de7183773e262ecdd5249d48f3ccebabe4ef958c7c328632187a6c8909b7f369cbc93dc423fea74e940478098fb00d3657a0c5431715eff9266712c111425149612c99120b07925b922aa3eaae003b8eb4c6fa4dd9570483bdcb6664e96f0c26132e2de75fbb708363f99962db18071360c47d035f9054605339affb1e98847e764054ba9e94e2446e13d1189371e043002bf619f64c602d6b1477ea1d496ab0eef5a612442e381d24ebe36263a867bc075fcde859d5a8398a0e56b8621c2f6c01d0aa74fce2128c3cf438e6c825fddcdaf29cf4e6d6bd131469b4fc796b331e4343da69054ad20ffacf4a0c4d6523341f66cbc629c9ef58bec92b1cefc1264627d1ae4509e3271702dfaf494ff07840393ed2e29f29df68cd9f219021a86952240caebf8725624fd9a7d1245c625d8fbced5ad0b3710dc1e57e20fcbbab8ce0bc57543ee99c1408d3aa48340f1553811ee4cbf70329eb028ac8a1c91a24f16075341729cdb1ccd0a5698ceec483eaa2d3d79eec0acf079912113c7bc9bf476fe33590268ff0fd45b36817bb0958939370185c1636dae3f43fea599b46d4ff4d6aee37bbccf415b68228612f69eddea178398c0c13d7f0a258b6719e2e0079b780a0863ff115b6d1962cf1c252ce3c7b50bd87442e40be31f1082cfacbcc570cc8fef44998e040f563d8521a40742d7f9e070644f29fbd119d41e437bc8307dba87c1a5f20ee54b07fdd0a7ddf7a322cc4c86194ed5f7ddfb2061feb8cdf1c92b38a50f7259a62814e8d73837c4aeb114954319a9b0a60ead3c7687c993af5c05ebc4098136eeaf6547c35b1a350a8bbb04ceeec682b16c1870836618e2133f32e9451b876e65e6e261bb99ee1460b8a7e1501f0f13bfff2204b1240b82c579987751cdecc9b4d77480e8781c275741c274f035ea2b79d0b43f27480c3a6c1221e010a8180f838a01ea72820a37982df204d7b7c19187e6b76a0e5a82d4b013b46aa9337308790572e8803f210bca8421db2c2dfb0e904911cd8de2d5163b83385f45f6667c3f52b0c23a36c96d964f363f1820006a3c02acb0cd696cc9b26a97050b7a57327d79a2c0c9ef6b23a4951979c317386d35186e5bf576b1b5c0d0b8a52446aff3da802f2a55f26df69b5eb2d1549a0bde209341152e920ba10bbe8386b7e3d787fe71414705c839ae24b2e98ef331e187a87c0c28a044e2563b3be0f2edd735b69253bf6002f83513e69aa6faca65222e73dab7da72149f9df950b72feeac33483e2f90bd740acbec8a1dd15bfebd369c29b52974cd8aa83ced04f086e1e93804a90c9ea682866203b74452fabac2859973154c3dd82f66fbbe5f733fee17abb37cf2d06354f5611e1616111f24fa78edf903077fbc8a41a4146928a23fbdf61d5eea0e3b13df32064e9bf61264ef755f62864280a024a2fd992bf8f7f963800af8fa4078e7ca3c8308dbea1aa329e5b856a7e58aad83efd0f3acc9854792b142f22d4e60f740e75decdfd380092f31c07f6dd9e30a3192e3caf485954fd0f18cf388213c6cc89b5ba1ae9b8dbd88bf3f82e004d3e1b6f1fec70c126f39a8a8b2a3159f536debc046d0502886e8bff9fd0a084fa4579876eb91fb5a5b2e9f3e7aabdbb1ef232fbee620ce64b6b15946c1bdeea29c96618771d1c943a1563f1badb3e24056b61ffe033a2f356441e00d554399e605c6988e572f7fb788d80fa813623332d14d40e665c57b85d7bfdcb87f451101250ec6f26652249d59dc974b7361d571a8101cdfd36aba3b5854d3ae086b5fdd4597721b66e3c0dc5d8c606d9657d0e323283a5217d1f53f2f284f57b85c8a61ac8924711f895c5ed90ef17745ed2d728abd22a5f7a13479a462d71b56c19a74a40b655c58edfe0a188ad2cf46cbf30524f65d423c837dd1ff2bf462ac4198007345bb44dbb7b1c861298cdf61982a833afc728fae1eda2f87aa2c9480858bec6260b1916e8e7a80fb12a60e60fddad614f7e9f6141b527b85080a199b347b20b984ccabda1a631a2f6fa8c309a10ce5fa0e4bdd3721ed9d416c1359c5f77083c0d896fae604737f45651b68262dc89834691eaee0d9b47d03ffd1bdc57673ffbf8596e75cf4b295851e86647d4814d70931edd0d5eaae0313e8cbab65588b7bfdb254d80291294536c2711fd79bfa8ac38e3dabdffee418b850898c6b46c1e0c3885936a284df78a2786401fbcd8199a8a798a7c5ba5ca23c5f94c9a0e71c14592f2c6a0274dc140f69e872b387b1ec01a5852c51c8208ae53c73c8cce4a40007ce55daeb0a2902c6bd84dac571936c15af8da249d038b9fc10aef3b7e4618c158e5df024eecd2b0f9bc1a1fa322eff2eee36717679aa16c802ceb128905ca1ca96cb1377c7d076312b262f1a738b8e928a0b01a06c8088933b5a69e9909c256f0f27aaaa3e3bca76973beceddbb0028aaa91bb8e51c2f9a3e3da85c08b29024c20fa1f86fa1511240488413a13a7415e7652050acf76504221d66c76290c1ea9f2cad9d3a46632849c9694cbc278355617d548da8ea675b61dd786d3134c63ceac224059e33715de87ec4e669129c4bab5a2edc737594f4700643b9ef1f1b7027787bcdf608fa27aed12d8d428c4d8d177425a997298c83ff206f3a16a04048f7b83b52fc276c19ff59198dd2e5058e4863cf918e66ac7e41a4416a2278b31bbdf9d930fb2f45a97c7f0f8484cd62c3b0a18310c3cd73580a1a60c6ad09126cfcc5ebe48d81255a88cb75fdf295d3c1e2a5f693c1be7f859aeb43dc10c1bcb12694c9fa2dbd79a1b6c6f35686eed70d50e980b712ca3f9b76af7126e833308cedff19ce4644c11388e9fe1acc28eb0a7df4b7e73171220e4fdf6f13bdee0c54955d51ed81a77a3461bcfcd73ce49e4dec5002a0a833c56278e3943eec52beacdf47e8e44a662368f3a8760853925f4c0e74077fe182a81c745497150df1a5fa0e37652f1af15b359145002a9ac2bc8966e92cd167fe38a0d4bade129d5b80c0a4b45d4a7de9603edd1b11ba9cf6248a7817f941b606f5a41c8edfd414e0ad95554717a2ddbef0c939833134b49d25d3d498e34d662d7d8abe3922c5ff31aafc72584ae4d4d28b892915c5358c383ed9eee47150c2e22eb92c523d54d6cef74aae706d7cddc47ee34e5b74197e47ee11397329f886c587d40962862d160911244c17d5c76315696bfbcc946d255ca22e46cd83b11c38275fb7d2a3ab478096fdedc73a710f45c977c3d1ccbcfbd6f975996c04f6edba7ba0ca2b0c5bd5b5e75bf606f1f63cd62dbb391373657c57627ba3d7245685dcf36bb2e426bb6b1980a3174fa2c36626f49acdfa2f03215e6bc4e887783c7f102d04f4968946524c0187a409b36535e8bc9ae50e6622854c73d5ea05673533e4970121aaba8945adc86e3eddaa58067be43020c8af3012a219132ad977bd3edabfe4aca920f09fbf73cb4a33fe28625119ff13a160a8441fda212d6c30511a531634080a6318db01622358953532d428bc5c10312117b3e625746436178d30fe2853c656d77e7214320aec89d613c63359eba8e076faccc5998151df93ce2d3c64d85a17147e1f5851e6687a5747d32151026047d8cdf91e94dc98c13b90698e5a3655b3a0a41d4e504a6eff4f5b65736044b8cfad0de351a5ea8b738570606487022bc4e29d06869bab628114ab07a9d63734f08a37061f4b4fbc7ef7eacc6f69a05eb370a56561b28de5984dd33425413e4742e6d5cbd803509e3c196be80f70b56107d179b49033766abd615075762fee624f5966bf0c5ca1c34350705db54a95d55a6b766e728cd95458e59f362cf0ac387e38665a3cb5c7ae2d3394980145e62e03c5180d892d3b44f3413c00ba321c1c482a5127d9bd16c14239452173f11acd9c74fb2c549320bf351aa36f9e72a2a045a5c39273c133af2ae81e747deaa8469cd18c25656d0e2576c861dd5a30d446ef111a448b6082ca5fb66d2b8f71032b8d8147832c09781a30d0f3d46ddb2582bd12d1989129a8a5592acd913952dfeed6fab9e028f6e2f00bc"))
                    (acl2::hex-string-to-bytes! "cfa7c3f12054db27ab677227b21948cd4aafec0b179027dffd033c87"))

(acl2::assert-equal (sha-224-bytes (acl2::hex-string-to-bytes! "f61363eaa08cce68283076fa1ba17ffeeade05f89e216891c703e7e0b5e671e9eb94b763d12486bfa91e5465c4df1941f63e607e476f9d91eb28c17a75030c46472422d94d1b62eaeb0b734f330c1f7b5ccf65456f81c9eb44f1ef56adf4689ff109ccd37c3909f3d441fa3fcff33cb404da3dd47f1d95a5b9a59afcc7822efbd18bdcd9c64524e14e0712be005b16f5d6a60fb3dd0ca9c9318ee9f529750663cff47b7393095abd40a4e3532e0b5545ba7d046bf315ac5e96f205c83d9597ded38ad02a4b4b548e91619486c1cca2ad98a544832802476dcf1a15fd69aa8a74c597048556d4b95c2b864edf2166a5345867f683e39a21fdc44e421d177b862d8d613e0d1c6c3230f55b38e7b7a105e2e20b42f38ea4e3165dd6b4afaed134121331a74806da6711d7277b25e9456abf1ed9515c90e934c082f9923b96743c2a404e67c07ae8ff07b9343e2120197ed4f4194b465131bd041b000449362c457efa02a5fde15ca31a83c8aa64c5e6e6284d125837394325e910333462f8c1f7142a938985a4fffa531d2ce7753e744dc30f0d9eb9b0e9157849da470d43fafcf6b94a3ce22999adb970a81cdcf58b59ae245248531ee935b8ae3d68d380a0f92c8d6fe69cdaf263546283712250f3b2f072297c8eeb0c2cb128a4bd3b157b99bc904aae391ed1c52ddc070fceb9730191e3da62f07f1aa36b9d566ecd6b5381b0a822bd435c007a4bc77a1e6897d5ee9a5eac240342db41eff79ca411eaf05c0fbf413b42fe452bb70925c57952df2012cae9e74f9d17af441cbe8790fa1ad0e2bced4fbc6b6980778c502d88525a400111d7d65bf6959472f6043ed104988530876f213b0b34209536437e1dfe0df5701abe16afb61e99826bb5f25c6e332501080fda46466db418f145dfc0c0c2e865cde8d2d3467a667b651bcabaa2b3555e2714a26684dee309a49f78faff3fc1eb129c30500c71a265cb7cef7206e0bf4dccc8a0212d4d566f379a7252244d2707d12158ac72caf5488521880c04810bd9df71fd02a983b552917cf55195b4c2c3f2c07372df4f092d7a52be110360e1074858e08e6048a7bb9c8d287dbb3c3b265dc1a927bbc6831b9457b9086b43e7e1efc5724fbb63d4b03f173461fead3a38fb6c21556df80e1bba77a079a3f3c86ebcffdbcc209ee53c62604182c7ce6602b20804835fad5034d4fb9bec15382ab69850428df8bf8264edb1766a7c68688b6bdf793af8adf16ecb2bd4ea28d9b6c269fecf1bac8f5232c952fba08743e7dc0f37a901a0bf9ad1854f70fa44a013991952acfda2d6feaddc8bf9ac4e278cfb8459bc60bfadf366ebaf9aa5d153e9438e94febb445c2ddd0fa498173bdcfff2b48dd9dfbd3da334e593787237bbe2c9b20cf0def9e888cb4be4ff67b429e7c9851b2807033853435c4232e9dfa0a4326b02a7ed2844b77e7695572723d8ed86e14ad25ff765aa9c3605f1922fcad786a3bd4e703b3661fe79fdc7dcefafb833af63ff70028e51465d24ddf6900024ce41860bef99872e543d6478108077f1d7d7aed6c08d79ef49b7e71d9c6b0c33112da506f3aaffc887a914914a3ef73f7b407380d9feef2dae2e95da0c26b4ab574f626903d665d49fa4543b1391f94b26d4fa2543beb8b37ca9783bcf2bc134a88f0459a88583fda5186b9ef2e143bce07a8aeb84369c28bb0fd6696ce0fa780dd5d500691f1fe8a0eca2e16f969f3bb11a6dfd26cd6334744af88b6baf291c8d091b86ce0c02409808412d1218d370a29de981b1ead480ae4126ca55ff14cb31131f135fb4f4ab965490a964cfa702bae947a0abf7b9b471dca84eb137842b00db29cda580bcd1a0b8d67eec24e59a02e7cf819e5bd878c503c53e07c141e4024981df27d6245cc11c5dd3327d53df807fc84d117ef972b39ec6c2e004139f3f466e6f6d1ff2f1fe686bd1bf43bb7a3e9a7c2b8dc8e64d4295f304c268c204c0b839a600c4ef3a6184499f1adfe312441e314499a166af7bd09429385515ba8c892bab672be3dff1583f67cbb7935098f530bfbd7140416438f15cc733d66aab9f8aa72abe9688b6cc944e5b3595dc4405c1c581a03daa08b352c3f96acbbb26e49ec1d2e4cfdcb9c9927e51c0b05d50382b7e6e9625926c2afe865562bd56043eb122e604588301357f1f35bfab7bc274b8110bbbc0a39a863ece638473e24d9709f1ee88f72696faa458909c4fdfa4a3aea7e078d58399ad5ac3ac36b70809d6ef5773d7756d81fe4952e9be6431fdac566776332eda71899a8cb7e14f7041e3ab7cfc5999ffc92bb635c88db218d677f87a545591512b0145520cee04c1fda6858711762d8576673228b571003d36ecdc070347c78fcdfca53c34c39ea34307c5a03a96f7a6919c065339d2174825f349370970983d62640073b6cae5c01deeb2f10021c10b21b17bdfdb18c0e0ac748932f037cb03ef209f1e1a6aa1bd9243959fea7f515b2e8ed7f424abc1add157e21f3f062a887e18ca66e90a9945aaeeb6d581186f3fc11b62b99bf4e2599e849d267f447b751634c973aa1496fe6ed75db94b55e55fe0406657c859ed891be3017137588771c05ce36ce5ef6090e5e6d273d29c7a6296e529f63343a8e74f5d5825efac1625837861792303bfa418045c2c9e56f5f1e8f2f4bc651bdd1b42d8e86d30a7f03967708bc4eab87be20d17450c35ec1fb3e04f0fd4e54752cdfc3dfb6f80b1dc54742ca972c5765bf1ce08d33a21c4c05185e4c2643b31d0a48675d7536a32a31eb39403ae39a5287a63c04fdfea48a072df995595f03f1e48e767c44273920d412b91e19ed0adaa24d226012216df86bff7570f8e8bdb066227fa0c64253ae262e4098a6a11f6a9982521302c2ff03bdfd4043a78946c4e9a4f18f6e00790ea2ce7e6473b84e8ed7d649b2524bde5887b7e8a1c7a935abf9765a8c9dea1fded443b0e6039b2ec3d7c957b29334625333bfecffdbd110c21027491ce887ceab255e86b1d8b2268633dc2a98281541804c65bbd276896ec3c8281bd4d041fbd0205f7e3c55a5cc62567f20c78356e26c5c59d3be1a4a861d7c096138d7a12e0a5fc6f60a2cde96c4ca22e8e55781db73cdfcd307299d1d3bc63ba7bd813deb963e358c9fa0d50c4a0987d165f71e73332bdf286236e98aab114f6dd74f60ab20462bb0616411934e8eee78de0495702700714243c4c854f7f7b5121aa4d1e314f209ffe3e92cd26ee4f74d91e27f28cbb643bf2055a2128ee3fd330c23da3a00dc60c9bba28f30178612de36234423ade7c70d8c5c1f39ec50984f004f0206606fb0ac4d12a132d4fcc1993f397ce729ff8babec6a8841ace806d4ab88e1deb0268a261ccc0b6123b3940d21b791e9dc880228c4e385a02cf4d9526253e2297c9b5db5bf31463180350ab862002bb241fbbaf2aa698ffbf9117046d9429b8d22289be988e4ba2a771b861984b592c6d6d52698016f2fbbadc87c3054d776604d78c5101e590fb274b1a6af1e4a7f9e279ec5877e56bd45a7745d9d8984fb595ae679e7e4009f7005a1dcf773313abea0d3285f55d2d14062cf84ecdbdd92ced70c2adf8e6de1d9a666a97147fc6c9ecefccf7e3c4604228c1482bcb033f5374e54484552038f55d37a8caaf12e8db52e861b62a632be727ffdf77bcb4a6b293d4962d7f363f08b29480713032a06983c099cbb082b78a6bae623a871638d2b00660b05c701addff469a937e2816fce73f1e3d5f692cb35c967c7c14721d3a35444913d615f2fdad169d38833c731be62d95141e5fb1eb627ed3464b0f7d728854f960d46feeee25c9326826b5df4fe851a959d4ad6eb8e5300f9f14377f0313fe199f3dc4373ace2a002c4fdd811f557400136b9c8ca54d72d99057304b743c0c51d6fb5df4680f00b08be6fa8213955e424aca723337020dd920fc430a4d6d44480ff5824e015212e98ed45509b310d71806d9778af86cc9b657ebb8e4170d70a751137b6d8cfe29543cff363f882fa72a5080f5ba6ff36d183d0cd70058845615a53d8f78917b0d989d50e44c1958951f2fa732ea2400c3a9db47e7879b39bf3b115b07979930865eb9afa957eef46b9fac367743a0f9910e388532a0659672c0a0552bf42b875c4e9905cb88906cef54ce150ced6a48a0f25cacfe744002394e571343b4cfe7560f443893088626eae39d2d390f0a53b8fb50a6942bb3e5da8a407e97020db83e4ac23249ff588c23f7216446d20984a0d11b593cec47a81e1e627c6f046e63daa61cd935958f05643663a37a5dee42221dc5af3edf82b1cdb2c5f612e2794ef3cc59179ff0b2f873486ad31fd941f40487578fe12def69afed72d5c9c5b7e2506381d3ce3b49cf19d63eda7b6d8b040fb6d8e06efe13388b7870ffdcd5ae87bb149df50b9cddf11e88e8a2d64252d90b53508c2d7978872bfac75e9339e8a6fa87b3649e3bf5c64a056f117a7f21dbc005c87a66c7a5bc551c80d9351aeda5d663ce0acf4119b665f6db11ae4d7d72f6754692610f2241d286a2570c9f43ee1751f3cee6d6d12936023589b9d4f006744bd5bc657a787c6adf35c25ef174e785f82c982a21a68786d28460e43b01ff032e7ae0bc302ed7e4d04201743708fae7018050e44b1f381d0dd08077ab39c56bc63fe6ac58ccb80a00cffc969a858e0713f8df5c7bc5a4c35c21d0ba4b1ec074bc2bb67e1bda89a218e79b0c42750aabd79a2bdada1e3440678038f4443a59e5b846397aa0e1b175bf8fc61d565fd89cf4701da690b44e5bd5f4a06fbd15f688619955e71069984b9ed16512fecc5fa213829d07c40f5cdd4f86f071df350123602bf30f92f8a6ec5cb4603364430784dcd694479fb5c48f7c3d1043d124e07f5c05fc384f18431fa2409fc8ad4c380cf676e8ac4830f0360521e92dbc6afd64fe2d56ab6e50d7ad187e0a19046f5a000a6a837fe708469d913a6176c88dbe9aeeadebb01828ca5f65758a6e8f35d529bfb84b07a6c3308bc99161681c178de8d38da18e752984fb322e050e7625124fbe6a2a8eabeb14c7542dd249e190d11756d3628facda6e4926af6442e2c8773aaf88075a1e075c1adc169a623e7ef1ef1e8915f9e6c4ee13ea029052fba46db13818c9b428a06f4c0655a3fb5c6b5738d4b6a2154a6d6d1f252f7086e8435d7e04245ae4f13db5257cf1d35c63de6aba1f63f02b015f5436803ceb25d22dad67b9b3a132a066966de6f214cc7a60923da897e98332220920a333eb79e0e8c73500ce86e65da47aebeb614d95c9783788165a7464063cf6a5351b57c83bfe11b0807cf147527dc968344566fd79aec16412b47918a537818ddffd165b0af7f59fe2c5ca9ea47bebac69a6464db28a999944aee1e5369b149f48e0127a7659d9cbcdb1f9337e26497f318cd7cc9b9525ef534f9214e1ee16a418cc326da7ae27fdea49674e060549a9340000c45d56ce2ee8a075ecd0baef65f9ca5e7bc47d46fbe194fa3764a054f11b09a97b1f55fc1daaa772b7288606a791e07eedf54f9e8d599a9b35c17fd3208eb0ba3a0a1b1885426e72bb14dbc53588197b0b64ab977dfa643392452b2cfe34fc1c49462f882e57fd5b42ba4093d6877a3b1d361999926df9a894172bc953b6f5b5f0fa19b6e0925a1792423dd851360da76b96210c935a62e484600710e7bba36e774f6208639b8f8cf3d55afd6f3cd9a73daa4c584968a5cdaa47486b82ec0f30256c91e87e19a4a4af9b2068f842e624da9a21e57c40cc4d4df57541ebf140e144792ebdfbb49f450dbb1682b4ef3d048b8f291cf38ade4bb69116f9eb713e6a1aa0c2efa0158a59545dc5b36aece53198b1ea6378ede4ae2e44b3fd4a1512b4f007d4e636e46cc7cdabb364e4e0e6914927ee23d111e560fb36d5b4d084c3a60dd6ec0984c84e77c0ca6afbe874490ee51abdc5c79fe948658e038644665927bb3692c2546a375174e572e23f50158fb6b1ddaf223ff94eb675c6db7065dd21404f4f796c722f3d3322517fb11b52fb0215d84baa981b5a3412a51b1a27567be548aa9495873f2343a9913c057d083c126d360027535954298e28ae0b54cc1df2492c7c97516debeda3b980052040392ee27d356017bb24db10d7fd0b9547efaf7c5b993a7ac37d793852b277a902da9e9c37541e8d48b393c13b6ee292d0d86307c25346a749c12f8fe33c24c35407a9c2ebc4295d08e798c8db6a57f26ec35812e613828206a057d1f485d41440a8987c22ed492e03a9344e3f5a335d2a95d4c6513fb7b3020038250ef54e44cda5b3f4633746a842e2ad7bf0a43e7d0b37204a8a41ade245cb1bb4b8f45cfc03e6424f360ae774cc784c7675b240f7199184c7197f2cb6d8e748a67aeac4e4eb548db823cef53aa955596856ea01765c8b47658763fcd7c23e0722dc3f4fa38cd5dc55b3eb39a1e25f1c781cb648a0dd0eb34b5a53e4ee0bb6a050675bc5e2ef52764e6dc978527e80dccf4e7ddf1da3000ea21db9aea3bc395a59a2a7ee481ab4a4ccdad07df5859fe39474a68c96b64ba19155fc209ddbae7f651589927e17e1aef780690f3100a377f0179b18b31fd5b4418c84038573fc559b496a782beec3dcf6e9faf5aef676e10bbec34b1be5888fda49b91e02890d2524c5b369f8a54175f29dedf8156fff690cf186ec77104a798315033bbd27c8362811ca2d8c4642c31222a8e00e7326561c384cc56ea905bc6477ca205d5415d2ceb6198c91d8b00c7f029575400bd3f2621c7d9ca9b6a09ea6f776968b19dc3f3e3b064eac64729960c51a7e543eee830724813acf420368711d6f65ef7d05c6a128fb3a87f170b2def1a1c5f1155f5232d9c16789521661ec721d55308879af1f065f19ca87929f21109110c0960ea70cc1a3ea7ff0c1d3407de92ef7421e42df5c9ab31d2ec0a750a9522869cbe4cabd66908d5823ec0494c639b7ebb4ddc70ab214d026efe21393454cd593bcc7026c50f116bdf47f3d11d676b9b5e59f85c791343890d1e02b146be7dee670a09841052c4e556962c6df3409fdfcdd4ac53f7f64b201dda237f8a38b0d0cef58d4650fabfdc98d7de72568584869a5c7cf99060c08211a107f6a76a028aa41d5fe00f6d4ebc065b8e80cfbcd65a444ffebfeae92009a90134b906a8ef86b6f015e394011dfe3fbfc10bb74cf1a54c2d96196fe5f63114791e29209bd45f47fc33c3977c9a3da562fa95a5bc53121283c78887edbf83148738d99db95d7f07c02df282263fd4c7e3bf913391970b57d1279fdb66e899f9aa294cd677f75056ba8902dca49b6e17c06490c9532a0b5c942ac1c2f81dc0645053f5535e1e9db569c9770f6e29642577ae06df77889cc14b4590e707abb0c5283d559cd24f91b44c63c1a2e966417177e085fe753f2a06750c8fa01c3ec59c806a003b8b16e6d3e0ec753fbcdea07c5f9070aee2ba4d9c9fe41096bc23796af75e6797def0f784b5c7f42e3efeebb67d526959368682043a60b6335e1666b0fb7f0e3a445a6e57398ad10706439db08f9ad285f72041ddc306f2fe7873f7ff1b6066ab51df95bbeaa99e42a95930ead4de4edd9b240779c84b5960a1b8c7b5c59703639ded283357a8be798e8c3fc60b91209898ee6d68b8a25bf250713065988756eb8eff705dff1cb4142c6934e282581a7f76fbce4b22b06d2c0b29dc9ef2b5e3ab8ba3088c5f5a019b098c9ac0e7977edacbe8c46184da2f96384f16c262b73c3c3a647b4ed6f54976bc42ebd189d3d032fe11ca7c5d93f5a97e5d996066efc31601b023361482de5fab48a2a50791b12084234adc2b8261de33d1bef98dd41ac18812baa29667428e5cea59abef65842c0479e0016d09f6bd0f48020e3608a5b714188c44bb3708bb57c46b669fa3ebade5a2149c90bf16ce572d0d1efe01e66f5c1e227d1166b33f1253726596e13a3eae28430e4506647aa20d3a9c54346f62c6a007bc276ab4b21303e91963b20246c47905c6e864425a76450516c734a339da4ac6927bd79f9cd0724a02947ad66edf00d766f97d42d65a71171ec567244d0d8ac9807d21d657e11b0fa197c0291f547c325f06a5e886b52dcb04ed605514d6cc9b26a9ddfb9bbb6ac476ec166cc0ab37912f7e72a5d41cb7ab699da4d849dcacb5c5906071a0d5ede3130fd233fb69cf9e58ff67fd5707cb32cd549900347587a40daf5dcb2fb71105e681a7f13806a0945bd7fa7187fcb4eba0b0d17f6d3809e06f852a7a90d5708761effc370f08672da8b6ae84df6221507420d2a86c9a9fe56b805b027cb798f225894367d8f56fb4d009f8b9c5ee8cebc823111dab23a63d624f63f53be3d6c0a20c12263e4436e462407d72e3d854280d361131c169d9531430e9c556d0c69d57d6ecd4978b6926c0088162b1e6139fc3e8579717b395c0d1d330f56604f4c2b78838058d1152e689b0b8ebb86e47bd8ca858c036b768013bb4edf7e206cdeb617e883b879c90be1fe798904ff76cfbb804991fa6dbcc8ea244855f434cde71b64861e90f83bf1802fb84d31107ffe4d66bad37e0a73fd3a6007b9e124d82f0567439908d4d2a7ed38b222684267d13787be99b5c803e7ad7875536c79f77d1a27083b4a97f5f7cf204d52d11b960d21e83e2aca32d54ee973796abeba919a02acbe4e7bc80ea7cff073249625747db29323cfe3cd4ffa9659fa47fe6f0d4f66fe4b18c6f0848094af0bdfdce86ae8938f53d02923dab35d33e2335521d562ee2699e25196bf4ac765fdccceebde156ce4b875c3bbdf40e6081874eb57509b412b8ef3444227cd0f1b313391b639cde49b3906d91dc46ad8e08452e799e6c408151495c87d24d015d88e5c50d1e4d8a3babdca95d2f3f9dbc30e17716fffa3eaec120e05cd1e1d80a4fe8e31c545fe524c2e0a1d61d2b599ba9e09b362030290b96fa722bd7d7af98769125c18c6079956730e9952cb65b7cc1c72d2f95aa7a18415a847d2a9b0288ab531fa55ceba1fe2741e322e44d3e002"))
                    (acl2::hex-string-to-bytes! "45ae4c1aa7ab5c5eb3ef6ed9bcd706fcfebf72bdb5f8f327e1f0bc02"))
