//***************************************************************************
//
//  Copyright (c) 1999 - 2006 Intel Corporation
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//***************************************************************************

/**
	@file	IFXScheduler.h

			This header file defines the IFXScheduler interface and its 
			functionality.
*/


#ifndef IFXScheduler_h
#define IFXScheduler_h


//***************************************************************************
//	Includes
//***************************************************************************


#include "IFXCoreServices.h"
#include "IFXNotificationManager.h"
#include "IFXSchedulerTypes.h"
#include "IFXSimulationManager.h"
#include "IFXSystemManager.h"
#include "IFXTimeManager.h"
#include "IFXUnknown.h"
#include "IFXSceneGraph.h"


//***************************************************************************
//	Interfaces and interface identifiers
//***************************************************************************


//---------------------------------------------------------------------------
/**
	This is the main interface for IFXScheduler.

	@note	The associated IID is named IID_IFXScheduler,
			and the CID for the component is CID_IFXScheduler.

	@note	To receive errors generated by tasks that are executed by the 
			Scheduler, register a task to handle IFXNotificationType_Error 
			notification types with the Notification Manager.  Please refer 
			to the IFXErrorInfo interface for additional details.
*/
class IFXScheduler : public IFXUnknown
{
public:

	/**
		This method retrieves the IFXNotificationManager.

		@param	ppComponent	Receives the IFXNotificationManager pointer.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppComponent is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetNotificationManager(IFXNotificationManager **ppComponent) = 0;

	/**
		This method retrieves the IFXTimeManager.

		@param	ppComponent	Receives the IFXTimeManager pointer.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppComponent is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetTimeManager(IFXTimeManager **ppComponent) = 0;

	/**
		This method retrieves the IFXSystemManager.

		@param	ppComponent	Receives the IFXSystemManager pointer.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppComponent is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSystemManager(IFXSystemManager **ppComponent) = 0;

	/**
		This method retrieves the IFXSimulationManager.

		@param	ppComponent	Receives the IFXSimulationManager pointer.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppComponent is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSimulationManager(IFXSimulationManager **ppComponent) = 0;

	/**
		This method advances the simulation manager to the specified time.

		@note	Advances the simulation. Does not run backwards. Does not 
				guarantee that the simulation will advance @e exactly to a 
				particular time, so you need to	query the simulation time 
				afterwards to find out the exact new time.

		@param	newTime	The new simulation time, in milliseconds.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  AdvanceSimulationTimeTo(U32 newTime) = 0;

	/**
		This convenience method retrieves the current frame from the SimulationManager.

		@param	pFrameNumber	Receives the current frame.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pFrameNumber is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSimulationFrame(U32 * pFrameNumber) = 0;

	/**
		This convenience method retrieves the simulation state from the SimulationManager.

		@param	pState	Receives the current frame.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pState is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSimulationState(BOOL *pState) = 0;

	/**
		This convenience method retrieves the current simulation time from the SimulationManager.

		@param	pTime	Receives the current simulation time.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTime is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSimulationTime(U32 *pTime) = 0;

	/**
		This convenience method retrieves the current simulation time delta from the SimulationManager.

		@param	pTime	Receives the current simulation time delta.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTime is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSimulationTimeDelta(U32 *pTime) = 0;

	/**
		This convenience method retrieves the current system time from the SimulationManager.

		@param	pTime	Receives the current system time.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTime is NULL.\n
	*/
	virtual IFXRESULT IFXAPI  GetSystemTime(U32 *pTime) = 0;

	/**
		This convenience method resets the simulation completely, removing all registered jobs
		and resetting all managers.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  ResetSimulation(void) = 0;

	/**
		This convenience method resets the simulation time to zero,
		but does not remove registered jobs. It is not recommended for client use.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  ResetSimulationTime(void) = 0;

	/**
		This convenience method pauses the simulation manager clock,
		flagging it for auto-restart upon the next Service.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  NotifyPause(void) = 0;

	/**
		This convenince method sets the simulation manager clock to the 
		desired state, and specifies the auto-reset flag.

		@param	running		If TRUE, the clock will be set to running.  If 
							FALSE, it will be set to stopped.
		@param	auto_reset	If TRUE, the clock's running state will toggle 
							on the next call to Service.  If FALSE, 
							it will not change.

		@return	Upon success, IFX_OK is returned.  Otherwise, 
				IFX_E_NOT_INITIALIZED is returned.
	*/
	virtual IFXRESULT IFXAPI  SetSimulationState(BOOL running, BOOL auto_reset = FALSE) = 0;

	/**
		Advances the SimulationManager clock to the specified Simulation Time.
		See IFXClock::AdvanceTo for more information.

		@param	time	The new time.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the component has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  AdvanceTo(U32 time) = 0;

	/**
		This method initializes the Scheduler component.

		@param	pCoreServices	A pointer to an IFXCoreServices interface.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_INVALID_POINTER is returned if pCoreServices is invalid.\n
		- @b IFX_E_ALREADY_INITIALIZED if the Scheduler has already been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  Initialize(IFXCoreServices * pCoreServices) = 0;

	/**
		This method retrieves an IFXCoreServices interface.

		@param	ppCoreServices	A pointer to a variable which will receive
		the IFXCoreServices interface.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_INVALID_POINTER if ppCoreServices is NULL.\n
		- @b IFX_E_NOT_INITIALIZED if the Scheduler has not yet been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  GetCoreServices(IFXCoreServices **ppCoreServices) = 0;

	/**
		This method causes the Scheduler to service any tasks which have been
		registered with its Task Manager.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Scheduler has not yet been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  Service() = 0;

	/**
		This method resets the Scheduler. The scheduler also sends a reset message
		to any tasks registered with it; tasks may unregister themselves as a result
		of a reset.

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Scheduler has not yet been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  Reset() = 0;

	/**
		This method retrieves the IFXTaskManager interface from the Scheduler's
		task manager.

		@param	ppTaskManager	A pointer to a variable which will receive
		the IFXTaskManager interface.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_INVALID_POINTER if ppTaskManager is NULL.\n
		- @b IFX_E_NOT_INITIALIZED if the Scheduler has not yet been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  GetTaskManager(IFXTaskManager ** ppTaskManager) = 0;

	/**
		This method returns the next available task handle. The Scheduler provides
		all task handles in the system in order to guarantee that each task handle
		is unique.

		@note	This method is used by IFXTaskManager and is not intended for client use.

		@param	pTaskHandle	A pointer to a variable which will receive the task handle.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_INVALID_POINTER if pTaskHandle is NULL.\n
		- @b IFX_E_NOT_INITIALIZED if the Scheduler has not yet been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  GetNextTaskHandle(IFXTaskHandle * pTaskHandle) = 0;
};

//---------------------------------------------------------------------------
/**
	This IID identifies the IFXScheduler interface.

	@note	The GUID string is {62a0a340-556c-11d3-a924-00a0c977eb2f}. 
*/
IFXDEFINE_GUID(IID_IFXScheduler,
0x62a0a340, 0x556c, 0x11d3, 0xa9, 0x24, 0x0, 0xa0, 0xc9, 0x77, 0xeb, 0x2f);


#endif


